<?php
/**
 * This file contains code related to the brands only.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WP_Term;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Brands' ) ) {
	/**
	 * Class Brands
	 */
	class Brands extends Singleton {
		const IMAGE_META_KEY = 'redparts_sputnik_image';

		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'init', array( $this, 'deferred_init' ) );
			add_action( 'current_screen', array( $this, 'current_screen' ) );
		}

		/**
		 * Deferred initialization.
		 */
		public function deferred_init() {
			$attribute_slug = $this->get_attribute_slug();

			if ( $attribute_slug ) {
				add_filter( 'manage_edit-' . $attribute_slug . '_columns', array( $this, 'brands_table_columns' ) );
				add_filter( 'manage_' . $attribute_slug . '_custom_column', array( $this, 'brands_table_image_column_content' ), 10, 3 );
				add_action( $attribute_slug . '_add_form_fields', array( $this, 'brand_add_form_image_field' ) );
				add_action( $attribute_slug . '_edit_form_fields', array( $this, 'brand_edit_form_image_field' ) );
				add_action( 'create_' . $attribute_slug, array( $this, 'save_brand_image' ) );
				add_action( 'edit_' . $attribute_slug, array( $this, 'save_brand_image' ) );
			}
		}

		/**
		 * Current screen hook handler.
		 */
		public function current_screen() {
			$attribute_slug = $this->get_attribute_slug();
			$current_screen = get_current_screen();

			if ( 'edit-' . $attribute_slug === $current_screen->id ) {
				wp_enqueue_media();
			}
		}

		/**
		 * Returns the brand attribute slug.
		 *
		 * @return null|string
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public function get_attribute_slug() {
			$attribute_name = Settings::instance()->get( 'brand_attribute' );

			if ( empty( $attribute_name ) ) {
				return null;
			}

			if ( ! function_exists( 'wc_attribute_taxonomy_name' ) ) {
				return null;
			}

			return wc_attribute_taxonomy_name( $attribute_name );
		}

		/**
		 * Returns the meta key of the brand image.
		 *
		 * @return string
		 */
		public function get_image_meta_key(): string {
			return self::IMAGE_META_KEY;
		}

		/**
		 * Returns columns of brands table.
		 *
		 * @param array $columns Columns array.
		 *
		 * @return array
		 */
		public function brands_table_columns( array $columns ): array {
			$cb_index = array_search( 'cb', array_keys( $columns ), true );

			if ( false === $cb_index ) {
				$columns[ self::IMAGE_META_KEY ] = esc_attr_x( 'Image', 'Admin', 'redparts-sputnik' );
			} else {
				$columns = array_slice( $columns, 0, $cb_index + 1, true )
					+ array( self::IMAGE_META_KEY => esc_attr_x( 'Image', 'Admin', 'redparts-sputnik' ) )
					+ array_slice( $columns, $cb_index + 1, null, true );
			}

			return $columns;
		}

		/**
		 * Outputs brand image.
		 *
		 * @param string $content     Column content.
		 * @param string $column_name Column name.
		 * @param int    $term_id     Term ID.
		 *
		 * @return string
		 */
		public function brands_table_image_column_content( string $content, string $column_name, int $term_id ): string {
			if ( self::IMAGE_META_KEY !== $column_name ) {
				return $content;
			}

			$image_id = $this->get_brand_image( absint( $term_id ) );
			$image    = wp_get_attachment_image( absint( $image_id ), array( 50, 50 ) );

			if ( ! $image ) {
				$image = wc_placeholder_img( array( 50, 50 ) );
			}

			return wp_kses( $image, 'redparts_sputnik_image' );
		}

		/**
		 * Outputs brand image form field.
		 *
		 * @param integer $attachment_id Image ID.
		 */
		public function the_image_field( $attachment_id = null ) {
			Admin::instance()->the_image_field( self::IMAGE_META_KEY, $attachment_id );
		}

		/**
		 * Adds a brand image field to the brand add form.
		 */
		public function brand_add_form_image_field() {
			?>
			<div class="form-field term-redparts-brand-image-wrap">
				<label><?php esc_html_e( 'Brand image', 'redparts-sputnik' ); ?></label>
				<?php $this->the_image_field(); ?>
			</div>
			<?php
		}

		/**
		 * Adds a brand image field to the brand edit form.
		 *
		 * @param WP_Term $term WordPress term object.
		 */
		public function brand_edit_form_image_field( WP_Term $term ) {
			$attachment_id = $this->get_brand_image( $term );

			?>
			<tr class="form-field term-redparts-brand-image-wrap">
				<th scope="row"><?php esc_html_e( 'Brand image', 'redparts-sputnik' ); ?></th>
				<td><?php $this->the_image_field( $attachment_id ); ?></td>
			</tr>
			<?php
		}

		/**
		 * Saves brand image.
		 *
		 * @param int $term_id Term ID.
		 */
		public function save_brand_image( int $term_id ) {
			if ( ! isset( $_POST[ self::IMAGE_META_KEY ] ) ) {
				return;
			}
			if ( ! current_user_can( 'edit_term', $term_id ) ) {
				return;
			}
			if ( ! isset( $_POST['_wpnonce'] ) && ! isset( $_POST['_wpnonce_add-tag'] ) ) {
				return;
			}

			// Sanitization here would be redundant.
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if (
				! wp_verify_nonce( wp_unslash( $_POST['_wpnonce_add-tag'] ), 'add-tag' )
				&& ! wp_verify_nonce( wp_unslash( $_POST['_wpnonce'] ), 'update-tag_' . $term_id )
			) {
				return;
			}
			// phpcs:enable

			$redparts_image = absint( $_POST[ self::IMAGE_META_KEY ] );

			if ( wp_attachment_is_image( $redparts_image ) ) {
				$this->update_brand_image( $term_id, $redparts_image );
			} else {
				$this->delete_brand_image( $term_id );
			}
		}

		/**
		 * Returns brand image.
		 *
		 * @since 1.6.0
		 *
		 * @param int|WP_Term $term Term ID or WP_Term object.
		 * @return mixed
		 */
		public function get_brand_image( $term ) {
			return get_term_meta( WPML::get_original_term_id( $term ), self::IMAGE_META_KEY, true );
		}

		/**
		 * Updates brand image.
		 *
		 * @since 1.6.0
		 *
		 * @param int|WP_Term $term  Term ID or WP_Term object.
		 * @param int         $image Image.
		 */
		public function update_brand_image( $term, int $image ) {
			update_term_meta( WPML::get_original_term_id( $term ), self::IMAGE_META_KEY, $image );
		}

		/**
		 * Delete brand image.
		 *
		 * @since 1.6.0
		 *
		 * @param int|WP_Term $term  Term ID or WP_Term object.
		 */
		public function delete_brand_image( $term ) {
			delete_term_meta( WPML::get_original_term_id( $term ), self::IMAGE_META_KEY );
		}
	}
}
