<?php
/**
 * Active filter shortcode.
 *
 * @package RedParts\Sputnik
 * @since 1.10.0
 */

namespace RedParts\Sputnik;

use Automattic\Jetpack\Constants;
use WC_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Filters' ) ) {
	/**
	 * Class Filters.
	 *
	 * @since 1.10.0
	 */
	class Filters extends Singleton {
		/**
		 * Initialization.
		 *
		 * @since 1.10.0
		 */
		public function init() {
			add_shortcode( 'redparts_sputnik_active_filters', array( $this, 'active_filters_shortcode' ) );

			add_action( 'redparts_sputnik_active_filters', array( $this, 'active_attribute_filters' ), 100 );
			add_action( 'redparts_sputnik_active_filters', array( $this, 'active_min_price_filter' ), 200 );
			add_action( 'redparts_sputnik_active_filters', array( $this, 'active_max_price_filter' ), 300 );
			add_action( 'redparts_sputnik_active_filters', array( $this, 'active_rating_filter' ), 400 );
			add_action( 'redparts_sputnik_active_filters', array( $this, 'active_vehicle_filter' ), 500 );

			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_attribute_args' ), 100 );
			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_min_price_arg' ), 200 );
			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_max_price_arg' ), 300 );
			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_rating_arg' ), 400 );
			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_vehicle_arg' ), 500 );
			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_orderby_arg' ), 600 );
			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_search_arg' ), 700 );
			add_filter( 'redparts_sputnik_filters_current_url', array( $this, 'current_url_post_type_arg' ), 800 );

			add_filter( 'woocommerce_widget_get_current_page_url', array( $this, 'woocommerce_get_current_url' ) );
		}

		/**
		 * Shortcode.
		 *
		 * @since 1.10.0
		 *
		 * @return false|string
		 */
		public function active_filters_shortcode() {
			if ( ! class_exists( 'WooCommerce' ) || ( ! is_shop() && ! is_product_taxonomy() ) ) {
				return '';
			}

			ob_start();

			echo '<ul>';

			ob_start();

			/**
			 * Hook: redparts_sputnik_active_filters.
			 *
			 * @hooked RedParts\Sputnik\Filters::active_attribute_filters - 100
			 * @hooked RedParts\Sputnik\Filters::active_min_price_filter  - 200
			 * @hooked RedParts\Sputnik\Filters::active_max_price_filter  - 300
			 * @hooked RedParts\Sputnik\Filters::active_rating_filter     - 400
			 * @hooked RedParts\Sputnik\Filters::active_vehicle_filter    - 500
			 */
			do_action( 'redparts_sputnik_active_filters' );

			// Attributes.

			$has_active_filters = ! empty( ob_get_contents() );

			ob_end_flush();

			echo '</ul>';

			if ( $has_active_filters ) {
				return ob_get_clean();
			}

			ob_end_clean();

			return '';
		}

		/**
		 * Outputs active attribute filters.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @since 1.10.0
		 */
		public function active_attribute_filters() {
			$base_link  = $this->get_current_url();
			$attributes = $this->get_chosen_attributes();

			if ( 0 >= count( $attributes ) ) {
				return;
			}

			foreach ( $attributes as $taxonomy => $data ) {
				foreach ( $data['terms'] as $term_slug ) {
					$term = get_term_by( 'slug', $term_slug, $taxonomy );

					if ( ! $term ) {
						continue;
					}

					// phpcs:disable WordPress.Security.NonceVerification.Recommended
					// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					$filter_name    = 'filter_' . wc_attribute_taxonomy_slug( $taxonomy );
					$current_filter = isset( $_GET[ $filter_name ] ) ? explode( ',', wc_clean( wp_unslash( $_GET[ $filter_name ] ) ) ) : array();
					$current_filter = array_map( 'sanitize_title', $current_filter );
					$new_filter     = array_diff( $current_filter, array( $term_slug ) );
					// phpcs:enable

					$link = remove_query_arg( array( 'add-to-cart', $filter_name ), $base_link );

					if ( count( $new_filter ) > 0 ) {
						$link = add_query_arg( $filter_name, implode( ',', $new_filter ), $link );
					}

					$this->remove_filter_link(
						$link,
						function() use ( $term ) {
							echo esc_html( $term->name );
						}
					);
				}
			}
		}

		/**
		 * Outputs active min price filter.
		 *
		 * @since 1.10.0
		 */
		public function active_min_price_filter() {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$base_link = $this->get_current_url();
			$min_price = isset( $_GET['min_price'] ) ? wc_clean( wp_unslash( $_GET['min_price'] ) ) : 0;

			if ( 0 >= $min_price ) {
				return;
			}

			$link = remove_query_arg( 'min_price', $base_link );

			$this->remove_filter_link(
				$link,
				function() use ( $min_price ) {
					echo wp_kses(
						/* translators: %s: minimum price */
						sprintf( __( 'Min %s', 'redparts-sputnik' ), wc_price( $min_price ) ),
						array()
					);
				}
			);
			// phpcs:enable
		}

		/**
		 * Outputs active max price filter.
		 *
		 * @since 1.10.0
		 */
		public function active_max_price_filter() {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$base_link = $this->get_current_url();
			$max_price = isset( $_GET['max_price'] ) ? wc_clean( wp_unslash( $_GET['max_price'] ) ) : 0;

			if ( 0 >= $max_price ) {
				return;
			}

			$link = remove_query_arg( 'max_price', $base_link );

			$this->remove_filter_link(
				$link,
				function() use ( $max_price ) {
					echo wp_kses(
						/* translators: %s: maximum price */
						sprintf( __( 'Max %s', 'redparts-sputnik' ), wc_price( $max_price ) ),
						array()
					);
				}
			);
			// phpcs:enable
		}

		/**
		 * Outputs active rating filter.
		 *
		 * @since 1.10.0
		 */
		public function active_rating_filter() {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$base_link     = $this->get_current_url();
			$rating_filter = isset( $_GET['rating_filter'] )
				? array_filter( array_map( 'absint', explode( ',', wp_unslash( $_GET['rating_filter'] ) ) ) )
				: array();

			if ( empty( $rating_filter ) ) {
				return;
			}

			foreach ( $rating_filter as $rating ) {
				$link_ratings = implode( ',', array_diff( $rating_filter, array( $rating ) ) );
				$link         = $link_ratings ? add_query_arg( 'rating_filter', $link_ratings ) : remove_query_arg( 'rating_filter', $base_link );

				$this->remove_filter_link(
					$link,
					function() use ( $rating ) {
						echo wp_kses(
							/* translators: %s: rating */
							sprintf( esc_html__( 'Rated %s out of 5', 'redparts-sputnik' ), esc_html( $rating ) ),
							array()
						);
					}
				);
			}
			// phpcs:enable
		}

		/**
		 * Outputs active vehicle filter.
		 *
		 * @since 1.10.0
		 */
		public function active_vehicle_filter() {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$base_link       = $this->get_current_url();
			$current_vehicle = Garage::instance()->get_current_vehicle( 'shop' );
			$checked         = isset( $_GET[ Vehicles::instance()->get_attribute_filter_name() ] );

			if ( ! $current_vehicle || ! $checked ) {
				return;
			}

			$query_arg_name  = Vehicles::instance()->get_attribute_filter_name();
			$query_arg_value = Vehicles::instance()->get_attribute_filter_value( $current_vehicle );

			$link = remove_query_arg( $query_arg_name, $base_link );
			$link = add_query_arg( array( 'current_' . $query_arg_name => $query_arg_value ), $link );

			$this->remove_filter_link(
				$link,
				function() use ( $current_vehicle ) {
					echo esc_html( Vehicles::get_vehicle_name( $current_vehicle ) );
				}
			);
			// phpcs:enable
		}

		/**
		 * Outputs remove filter link.
		 *
		 * @since 1.10.0
		 *
		 * @param string   $link           - Link.
		 * @param callable $label_callback - Function that outputs label.
		 */
		public function remove_filter_link( string $link, callable $label_callback ) {
			?>
			<li class="chosen">
				<a
					rel="nofollow"
					aria-label="<?php echo esc_attr__( 'Remove filter', 'redparts-sputnik' ); ?>"
					href="<?php echo esc_url( $link ); ?>"
				>
					<?php $label_callback(); ?>
				</a>
			</li>
			<?php
		}

		/**
		 * Functional copy of the WC_Widget::get_current_page_url.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @since 1.10.0
		 *
		 * @return string
		 */
		public function get_current_url(): string {
			static $link = false;

			if ( false !== $link ) {
				return $link;
			}

			if ( Constants::is_defined( 'SHOP_IS_ON_FRONT' ) ) {
				$link = home_url();
			} elseif ( is_shop() ) {
				$link = get_permalink( wc_get_page_id( 'shop' ) );
			} elseif ( is_product_category() ) {
				$link = get_term_link( get_query_var( 'product_cat' ), 'product_cat' );
			} elseif ( is_product_tag() ) {
				$link = get_term_link( get_query_var( 'product_tag' ), 'product_tag' );
			} else {
				$queried_object = get_queried_object();
				$link           = get_term_link( $queried_object->slug, $queried_object->taxonomy );
			}

			/**
			 * Hook: redparts_sputnik_filters_current_url.
			 *
			 * @hooked RedParts\Sputnik\Filters::current_url_attribute_args - 100
			 * @hooked RedParts\Sputnik\Filters::current_url_min_price_arg  - 200
			 * @hooked RedParts\Sputnik\Filters::current_url_max_price_arg  - 300
			 * @hooked RedParts\Sputnik\Filters::current_url_rating_arg     - 400
			 * @hooked RedParts\Sputnik\Filters::current_url_vehicle_arg    - 500
			 * @hooked RedParts\Sputnik\Filters::current_url_orderby_arg    - 600
			 * @hooked RedParts\Sputnik\Filters::current_url_search_arg     - 700
			 * @hooked RedParts\Sputnik\Filters::current_url_post_type_arg  - 800
			 *
			 * @noinspection PhpUnnecessaryLocalVariableInspection
			 */
			$link = apply_filters( 'redparts_sputnik_filters_current_url', $link );

			return $link;
		}

		/**
		 * Overrides standard WooCommerce WC_Widget::get_current_page_url function.
		 *
		 * @since 1.10.0
		 *
		 * @return string
		 */
		public function woocommerce_get_current_url(): string {
			return $this->get_current_url();
		}

		/**
		 * Adds attribute args to the link.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_attribute_args( string $link ): string {
			$attributes = $this->get_chosen_attributes();

			if ( $attributes ) {
				foreach ( $attributes as $name => $data ) {
					$filter_name = wc_attribute_taxonomy_slug( $name );

					if ( ! empty( $data['terms'] ) ) {
						$link = add_query_arg( 'filter_' . $filter_name, implode( ',', $data['terms'] ), $link );
					}
					if ( 'or' === $data['query_type'] ) {
						$link = add_query_arg( 'query_type_' . $filter_name, 'or', $link );
					}
				}
			}

			return $link;
		}

		/**
		 * Adds min price arg to the link.
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_min_price_arg( string $link ): string {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( isset( $_GET['min_price'] ) ) {
				$link = add_query_arg( 'min_price', wc_clean( wp_unslash( $_GET['min_price'] ) ), $link );
			}

			return $link;
			// phpcs:enable
		}

		/**
		 * Adds max price arg to the link.
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_max_price_arg( string $link ): string {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( isset( $_GET['max_price'] ) ) {
				$link = add_query_arg( 'max_price', wc_clean( wp_unslash( $_GET['max_price'] ) ), $link );
			}

			return $link;
			// phpcs:enable
		}

		/**
		 * Adds rating arg to the link.
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_rating_arg( string $link ): string {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( isset( $_GET['rating_filter'] ) ) {
				$link = add_query_arg( 'rating_filter', wc_clean( wp_unslash( $_GET['rating_filter'] ) ), $link );
			}

			return $link;
			// phpcs:enable
		}

		/**
		 * Adds vehicle arg to the link.
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_vehicle_arg( string $link ): string {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$current_vehicle = Garage::instance()->get_current_vehicle( 'shop' );
			$checked         = $current_vehicle && isset( $_GET[ Vehicles::instance()->get_attribute_filter_name() ] );

			if ( $current_vehicle ) {
				$query_arg_name  = Vehicles::instance()->get_attribute_filter_name();
				$query_arg_value = Vehicles::instance()->get_attribute_filter_value( $current_vehicle );

				$link = remove_query_arg( $query_arg_name, $link );

				if ( $checked ) {
					$link = add_query_arg( array( $query_arg_name => $query_arg_value ), $link );
				} else {
					$link = add_query_arg( array( 'current_' . $query_arg_name => $query_arg_value ), $link );
				}
			}

			return $link;
			// phpcs:enable
		}

		/**
		 * Adds orderby arg to the link.
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_orderby_arg( string $link ): string {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( isset( $_GET['orderby'] ) ) {
				$link = add_query_arg( 'orderby', wc_clean( wp_unslash( $_GET['orderby'] ) ), $link );
			}

			return $link;
			// phpcs:enable
		}

		/**
		 * Adds search arg to the link.
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_search_arg( string $link ): string {
			// To support quote characters, first they are decoded from &quot; entities, then URL encoded.
			if ( get_search_query() ) {
				$link = add_query_arg( 's', rawurlencode( htmlspecialchars_decode( get_search_query() ) ), $link );
			}

			return $link;
		}

		/**
		 * Adds post type arg to the link.
		 *
		 * @since 1.10.0
		 *
		 * @param string $link Current page URL.
		 *
		 * @return string
		 */
		public function current_url_post_type_arg( string $link ): string {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( isset( $_GET['post_type'] ) ) {
				$link = add_query_arg( 'post_type', wc_clean( wp_unslash( $_GET['post_type'] ) ), $link );

				// Prevent post type and page id when pretty permalinks are disabled.
				if ( is_shop() ) {
					$link = remove_query_arg( 'page_id', $link );
				}
			}

			return $link;
			// phpcs:enable
		}

		/**
		 * Get an array of attributes and terms selected with the layered nav widget.
		 *
		 * @since 1.10.0
		 *
		 * @return array
		 */
		public function get_chosen_attributes(): array {
			static $attributes = null;

			if ( ! is_array( $attributes ) ) {
				$attributes      = WC_Query::get_layered_nav_chosen_attributes();
				$current_vehicle = Garage::instance()->get_current_vehicle( 'shop' );
				$vehicle_slugs   = $current_vehicle ? (array) $current_vehicle['slug'] : array();

				foreach ( $attributes as $taxonomy => $data ) {
					if ( Vehicles::instance()->get_attribute_slug() !== $taxonomy ) {
						continue;
					}

					$attributes[ $taxonomy ]['terms'] = array_filter(
						$data['terms'],
						function( $term_slug ) use ( $vehicle_slugs ) {
							return ! in_array( $term_slug, $vehicle_slugs, true );
						}
					);
				}

				$attributes = array_filter(
					$attributes,
					function( $attribute ) {
						return ! empty( $attribute['terms'] );
					}
				);
			}

			return $attributes;
		}
	}
}
