<?php
/**
 * RedParts sputnik settings.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use Redux;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Settings' ) ) {
	/**
	 * Class Settings.
	 *
	 * @package RedParts\Sputnik
	 */
	class Settings extends Singleton {
		const REDUX_OPTION_NAME = 'redparts_sputnik_settings';

		/**
		 * Settings default values.
		 *
		 * @var array
		 */
		protected static $default_values = array();

		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'plugins_loaded', array( $this, 'redux_init' ) );

			self::$default_values = array(
				'compare_enabled'                  => 'yes',
				'compare_rows'                     => implode(
					', ',
					array(
						'rating',
						'availability',
						'price',
						'add_to_cart',
						'sku',
						'weight',
						'dimensions',
						'attributes',
					)
				),
				'wishlist_enabled'                 => 'yes',
				'global_featured_attributes'       => array(),
				'autoparts_features'               => 'yes',
				'garage_page'                      => 'yes',
				'search_vehicles_by_vin'           => 'yes',
				'vehicle_fields'                   => array(
					array(
						'type'        => 'year',
						'slug'        => 'produced',
						'label'       => esc_html__( 'Year', 'redparts-sputnik' ),
						'placeholder' => esc_html__( 'Select Year', 'redparts-sputnik' ),
					),
					array(
						'type'        => 'text',
						'slug'        => 'brand',
						'label'       => esc_html__( 'Brand', 'redparts-sputnik' ),
						'placeholder' => esc_html__( 'Select Brand', 'redparts-sputnik' ),
					),
					array(
						'type'        => 'text',
						'slug'        => 'model',
						'label'       => esc_html__( 'Model', 'redparts-sputnik' ),
						'placeholder' => esc_html__( 'Select Model', 'redparts-sputnik' ),
					),
					array(
						'type'        => 'text',
						'slug'        => 'modification',
						'label'       => esc_html__( 'Engine', 'redparts-sputnik' ),
						'placeholder' => esc_html__( 'Select Engine', 'redparts-sputnik' ),
					),
				),
				'vehicle_name'                     => '%produced_since% %brand% %model%',
				'vehicle_description'              => '%modification%',
				'vehicle_picker_group_same_values' => 'yes',
				'search_by_sku'                    => 'yes',
				'search_by_sku_match'              => 'exactly',
			);
		}

		/**
		 * Redux initialization.
		 *
		 * @noinspection PhpUnused
		 */
		public function redux_init() {
			if ( ! class_exists( 'Redux' ) ) {
				return;
			}

			$plugin_data = array(
				'Name'    => 'RedParts Sputnik',
				'Version' => '',
			);

			if ( function_exists( 'get_plugin_data' ) ) {
				$plugin_data = get_plugin_data( RED_PARTS_SPUTNIK_FILE );
			}

			add_filter( 'redux/options/' . self::REDUX_OPTION_NAME . '/sections', array( $this, 'redux_dynamic_sections' ) );
			add_filter( 'redux/' . self::REDUX_OPTION_NAME . '/field/class/redparts_vehicle_fields', array( $this, 'redux_vehicle_fields_file' ) );
			add_filter( 'redux/' . self::REDUX_OPTION_NAME . '/field/class/redparts_vehicle_import_export', array( $this, 'redux_vehicle_import_export_file' ) );

			$args = array(
				// This is where your data is stored in the database and also becomes your global variable name.
				'opt_name'             => self::REDUX_OPTION_NAME,
				// Name that appears at the top of your panel.
				'display_name'         => $plugin_data['Name'],
				// Version that appears at the top of your panel.
				'display_version'      => $plugin_data['Version'],
				// Specify if the admin menu should appear or not. Options: menu or submenu (Under appearance only).
				'menu_type'            => 'menu',
				// Show the sections below the admin menu item or not.
				'allow_sub_menu'       => true,
				'menu_title'           => esc_html__( 'RedParts Sputnik', 'redparts-sputnik' ),
				'page_title'           => esc_html__( 'RedParts Sputnik Settings', 'redparts-sputnik' ),

				// You will need to generate a Google API key to use this feature.
				'google_api_key'       => '',
				// Set it you want google fonts to update weekly. A google_api_key value is required.
				'google_update_weekly' => false,
				// Must be defined to add google fonts to the typography module
				// Use a asynchronous font on the front end or font string.
				'async_typography'     => false,

				// Show the panel pages on the admin bar.
				'admin_bar'            => false,
				// Choose an icon for the admin bar menu.
				'admin_bar_icon'       => 'dashicons-portfolio',
				// Choose an priority for the admin bar menu.
				'admin_bar_priority'   => 50,

				// Set a different name for your global variable other than the opt_name.
				'global_variable'      => '',
				// Show the time the page took to load, etc.
				'dev_mode'             => false,
				// If dev_mode is enabled, will notify developer of updated versions available in the GitHub Repo.
				'update_notice'        => false,

				// Enable basic customizer support.
				'customizer'           => true,

				// Order where the menu appears in the admin area. If there is any conflict, something will not show. Warning.
				'page_priority'        => null,
				// For a full list of options, visit: http://codex.wordpress.org/Function_Reference/add_submenu_page#Parameters.
				'page_parent'          => 'themes.php',
				// Permissions needed to access the options panel.
				'page_permissions'     => 'manage_options',
				// Specify a custom URL to an icon.
				'menu_icon'            => '',
				// Force your panel to always open to a specific tab (by id).
				'last_tab'             => '',
				// Icon displayed in the admin panel next to your menu_title.
				'page_icon'            => 'icon-themes',
				// Page slug used to denote the panel, will be based off page title then menu title then opt_name if not provided.
				'page_slug'            => 'redparts-sputnik-settings',
				// On load save the defaults to DB before user clicks save or not.
				'save_defaults'        => true,
				// If true, shows the default value next to each field that is not the default value.
				'default_show'         => false,
				// What to print by the field's title if the value shown is default. Suggested: *.
				'default_mark'         => '',
				// Shows the Import/Export panel when not used as a field.
				'show_import_export'   => true,

				// CAREFUL -> These options are for advanced use only.
				'transient_time'       => 60 * MINUTE_IN_SECONDS,
				// Global shut-off for dynamic CSS output by the framework. Will also disable google fonts output.
				'output'               => true,
				// Allows dynamic CSS to be generated for customizer and google fonts, but stops the dynamic CSS from going to the head.
				'output_tag'           => true,

				// FUTURE -> Not in use yet, but reserved or partially implemented. Use at your own risk.
				// possible: options, theme_mods, theme_mods_expanded, transient. Not fully functional, warning!
				'database'             => '',
				// If you prefer not to use the CDN for Select2, Ace Editor, and others, you may download the Redux Vendor Support plugin
				// yourself and run locally or embed it in your code.
				'use_cdn'              => true,
			);

			Redux::set_args( self::REDUX_OPTION_NAME, $args );

			Redux::set_section(
				self::REDUX_OPTION_NAME,
				array(
					'title' => esc_html__( 'Noop', 'redparts-sputnik' ),
					'id'    => 'noop',
				)
			);
		}

		/**
		 * Adds dynamic section the redux plugin.
		 *
		 * @param array $sections Sections array.
		 * @return array
		 */
		public function redux_dynamic_sections( array $sections ): array {
			if ( ! class_exists( 'RedParts\Sputnik\Share_Buttons' ) ) {
				return $sections;
			}

			$sections = array_filter(
				$sections,
				function ( $section ) {
					return 'noop' !== $section['id'];
				}
			);

			$sections[] = array(
				'title'  => esc_html__( 'Social Networks', 'redparts-sputnik' ),
				'id'     => 'social-networks',
				'icon'   => 'el el-group',
				'fields' => array(
					array(
						'id'    => 'social_links_menu',
						'type'  => 'select',
						'data'  => 'menus',
						'title' => esc_html__( 'Social Links Menu', 'redparts-sputnik' ),
					),
					array(
						'id'    => 'share_buttons_menu',
						'type'  => 'select',
						'data'  => 'menus',
						'title' => esc_html__( 'Share Buttons Menu', 'redparts-sputnik' ),
					),
				),
			);

			$sections[] = array(
				'title'  => esc_html__( 'Compare', 'redparts-sputnik' ),
				'id'     => 'compare',
				'icon'   => 'el el-refresh',
				'fields' => array(
					array(
						'id'      => 'compare_enabled',
						'type'    => 'select',
						'title'   => esc_html__( 'Enabled', 'redparts-sputnik' ),
						'options' => array(
							'yes' => 'Yes',
							'no'  => 'No',
						),
						'default' => self::$default_values['compare_enabled'],
					),
					array(
						'id'       => 'compare_page',
						'type'     => 'select',
						'data'     => 'pages',
						'title'    => esc_html__( 'Compare Page', 'redparts-sputnik' ),
						'required' => array( 'compare_enabled', '=', 'yes' ),
					),
					array(
						'id'          => 'compare_rows',
						'type'        => 'textarea',
						'title'       => esc_html__( 'Rows', 'redparts-sputnik' ),
						'subtitle'    => esc_html__( 'Comma-separated names of the rows to be displayed in the comparison table.', 'redparts-sputnik' ),
						// translators: %s list of possible values.
						'description' => sprintf( esc_html__( 'Possible values: %s', 'redparts-sputnik' ), self::$default_values['compare_rows'] ),
						'rows'        => 3,
						'default'     => self::$default_values['compare_rows'],
						'required'    => array( 'compare_enabled', '=', 'yes' ),
					),
					array(
						'id'       => 'compare_excluded_attributes',
						'type'     => 'textarea',
						'title'    => esc_html__( 'Excluded attributes', 'redparts-sputnik' ),
						'subtitle' => esc_html__( 'A comma-separated list of product attribute slugs that should not appear in the comparison table.', 'redparts-sputnik' ),
						'rows'     => 3,
						'default'  => '',
						'required' => array( 'compare_enabled', '=', 'yes' ),
					),
				),
			);

			$sections[] = array(
				'title'  => esc_html__( 'Wishlist', 'redparts-sputnik' ),
				'id'     => 'wishlist',
				'icon'   => 'el el-refresh',
				'fields' => array(
					array(
						'id'      => 'wishlist_enabled',
						'type'    => 'select',
						'title'   => esc_html__( 'Enabled', 'redparts-sputnik' ),
						'options' => array(
							'yes' => 'Yes',
							'no'  => 'No',
						),
						'default' => self::$default_values['wishlist_enabled'],
					),
					array(
						'id'       => 'wishlist_page',
						'type'     => 'select',
						'data'     => 'pages',
						'title'    => esc_html__( 'Wishlist Page', 'redparts-sputnik' ),
						'required' => array( 'wishlist_enabled', '=', 'yes' ),
					),
				),
			);

			if ( class_exists( 'WooCommerce' ) ) {
				$attributes = array();

				foreach ( wc_get_attribute_taxonomies() as $attribute ) {
					$attributes[ $attribute->attribute_name ] = $attribute->attribute_label;
				}

				$sections[] = array(
					'title'  => esc_html__( 'Brands', 'redparts-sputnik' ),
					'id'     => 'brands',
					'icon'   => 'el el-globe',
					'fields' => array(
						array(
							'id'      => 'brand_attribute',
							'type'    => 'select',
							'title'   => esc_html__( 'Brand Attribute', 'redparts-sputnik' ),
							'options' => $attributes,
						),
					),
				);

				$sections[] = array(
					'title'  => esc_html__( 'Featured Attributes', 'redparts-sputnik' ),
					'id'     => 'featured-attributes',
					'icon'   => 'el el-th-list',
					'fields' => array(
						array(
							'id'      => 'global_featured_attributes',
							'type'    => 'select',
							'multi'   => true,
							'title'   => esc_html__( 'Global featured attributes', 'redparts-sputnik' ),
							'options' => $attributes,
							'default' => self::$default_values['global_featured_attributes'],
						),
					),
				);

				$sections[] = array(
					'title'  => esc_html__( 'Auto Parts', 'redparts-sputnik' ),
					'id'     => 'autoparts',
					'icon'   => 'el el-car',
					'fields' => array(
						array(
							'id'       => 'autoparts_features',
							'type'     => 'select',
							'title'    => esc_html__( 'Use auto parts store features', 'redparts-sputnik' ),
							'subtitle' => esc_html__( 'Garage, parts finder block, compatibility badge, etc. Set to "no" if you are not going to sell auto parts.', 'redparts-sputnik' ),
							'options'  => array(
								'yes' => 'Yes',
								'no'  => 'No',
							),
							'default'  => self::$default_values['autoparts_features'],
						),
						array(
							'id'       => 'garage_page',
							'type'     => 'select',
							'title'    => esc_html__( 'Show garage page in my account', 'redparts-sputnik' ),
							'options'  => array(
								'yes' => 'Yes',
								'no'  => 'No',
							),
							'default'  => self::$default_values['garage_page'],
							'required' => array( 'autoparts_features', '=', 'yes' ),
						),
						array(
							'id'       => 'vehicle_attribute',
							'type'     => 'select',
							'title'    => esc_html__( 'Vehicle Attribute', 'redparts-sputnik' ),
							'options'  => $attributes,
							'required' => array( 'autoparts_features', '=', 'yes' ),
						),
						array(
							'id'       => 'search_vehicles_by_vin',
							'type'     => 'select',
							'title'    => esc_html__( 'Search vehicles by VIN', 'redparts-sputnik' ),
							'subtitle' => esc_html__( 'Enables or disables the ability to search for vehicles by VIN.', 'redparts-sputnik' ),
							'options'  => array(
								'yes' => 'Yes',
								'no'  => 'No',
							),
							'default'  => self::$default_values['search_vehicles_by_vin'],
							'required' => array( 'autoparts_features', '=', 'yes' ),
						),
						array(
							'id'                => 'vehicle_fields',
							'type'              => 'redparts_vehicle_fields',
							'title'             => esc_html__( 'Vehicle Fields', 'redparts-sputnik' ),
							'required'          => array( 'autoparts_features', '=', 'yes' ),
							'validate_callback' => array( $this, 'vehicle_fields_validator' ),
							'default'           => self::$default_values['vehicle_fields'],
						),
						array(
							'id'       => 'vehicle_name',
							'type'     => 'text',
							'title'    => esc_html__( 'Vehicle Name', 'redparts-sputnik' ),
							'default'  => self::$default_values['vehicle_name'],
							'required' => array( 'autoparts_features', '=', 'yes' ),
						),
						array(
							'id'       => 'vehicle_description',
							'type'     => 'text',
							'title'    => esc_html__( 'Vehicle Description', 'redparts-sputnik' ),
							'default'  => self::$default_values['vehicle_description'],
							'required' => array( 'autoparts_features', '=', 'yes' ),
						),
						array(
							'id'       => 'vehicle_import_export',
							'type'     => 'redparts_vehicle_import_export',
							'title'    => esc_html__( 'Import and export', 'redparts-sputnik' ),
							'required' => array( 'autoparts_features', '=', 'yes' ),
						),
						array(
							'id'       => 'vehicle_picker_group_same_values',
							'type'     => 'select',
							'title'    => esc_html__( 'Group duplicate values in the last field of the vehicle picker', 'redparts-sputnik' ),
							'subtitle' => esc_html__( 'Deprecated. Should always be "Yes".', 'redparts-sputnik' ),
							'options'  => array(
								'yes' => 'Yes',
								'no'  => 'No',
							),
							'default'  => self::$default_values['vehicle_picker_group_same_values'],
							'required' => array( 'autoparts_features', '=', 'yes' ),
						),
					),
				);

				$sections[] = array(
					'title'  => esc_html__( 'Search', 'redparts-sputnik' ),
					'desc'   => esc_html__( 'The search enhancements presented in this section do not currently use a specialized index table, so they can significantly slow down the search. Use these opportunities at your own risk.', 'redparts-sputnik' ),
					'id'     => 'search',
					'icon'   => 'el el-search',
					'fields' => array(
						array(
							'id'      => 'search_by_sku',
							'type'    => 'select',
							'title'   => esc_html__( 'Search products by SKU', 'redparts-sputnik' ),
							'options' => array(
								'yes' => 'Yes',
								'no'  => 'No',
							),
							'default' => self::$default_values['search_by_sku'],
						),
						array(
							'id'       => 'search_by_sku_match',
							'type'     => 'select',
							'title'    => esc_html__( 'Search products by SKU (match type)', 'redparts-sputnik' ),
							'options'  => array(
								'exactly'  => 'Exactly matches the search term',
								'starts'   => 'Starts with the search term',
								'contains' => 'Contains the search term',
							),
							'default'  => self::$default_values['search_by_sku_match'],
							'required' => array( 'search_by_sku', '=', 'yes' ),
						),
						array(
							'id'       => 'search_by_attributes',
							'type'     => 'select',
							'title'    => esc_html__( 'Search products by attributes (name exactly matches the search term)', 'redparts-sputnik' ),
							'subtitle' => esc_html__( 'Select the attributes that will be included in the search. Only those attributes will be found whose name exactly matches the search term.', 'redparts-sputnik' ),
							'options'  => $attributes,
							'multi'    => true,
						),
						array(
							'id'       => 'search_by_attributes_starts',
							'type'     => 'select',
							'title'    => esc_html__( 'Search products by attributes (name starts with the search term)', 'redparts-sputnik' ),
							'subtitle' => esc_html__( 'Select the attributes that will be included in the search. Only those attributes will be found whose name starts with the search term.', 'redparts-sputnik' ),
							'options'  => $attributes,
							'multi'    => true,
						),
						array(
							'id'       => 'search_by_attributes_contains',
							'type'     => 'select',
							'title'    => esc_html__( 'Search products by attributes (name contains the search term)', 'redparts-sputnik' ),
							'subtitle' => esc_html__( 'Select the attributes that will be included in the search. Only those attributes will be found whose name contains the search term.', 'redparts-sputnik' ),
							'options'  => $attributes,
							'multi'    => true,
						),
					),
				);
			}

			return $sections;
		}

		/**
		 * Returns all settings.
		 *
		 * @return array
		 */
		public function all(): array {
			if ( empty( $GLOBALS[ self::REDUX_OPTION_NAME ] ) ) {
				return array();
			}

			return $GLOBALS[ self::REDUX_OPTION_NAME ];
		}

		/**
		 * Returns option value by key.
		 *
		 * @param string $key           - Option key.
		 * @param null   $default_value - Default option value.
		 *
		 * @return null
		 */
		public function get( string $key, $default_value = null ) {
			$settings = $this->all();

			if ( ! isset( $settings[ $key ] ) && isset( self::$default_values[ $key ] ) ) {
				return self::$default_values[ $key ];
			} elseif ( empty( $settings[ $key ] ) ) {
				return $default_value;
			}

			return $settings[ $key ];
		}

		/**
		 * Returns path to the redux vehicle fields.
		 *
		 * @return string
		 */
		public function redux_vehicle_fields_file(): string {
			return dirname( __FILE__ ) . '/redux/vehicle-fields/class-vehicle-fields.php';
		}

		/**
		 * Returns path to the redux vehicle import export.
		 *
		 * @return string
		 */
		public function redux_vehicle_import_export_file(): string {
			return dirname( __FILE__ ) . '/redux/vehicle-import-export/class-vehicle-import-export.php';
		}

		/**
		 * Custom function for the callback validation referenced above.
		 *
		 * @noinspection PhpUnused
		 *
		 * @param array $field          Field array.
		 * @param mixed $value          New value.
		 * @param mixed $existing_value Existing value.
		 *
		 * @return array
		 */
		public function vehicle_fields_validator( array $field, $value, $existing_value ): array {
			$error   = false;
			$message = '';

			$set_error = function( $error_message = '' ) use ( &$error, &$message, &$value, &$existing_value ) {
				if ( false === $error ) {
					$error   = true;
					$value   = $existing_value;
					$message = $error_message;
				}
			};

			if ( ! is_array( $value ) ) {
				$set_error();
			} else {
				$value     = array_values( $value );
				$all_slugs = array();

				foreach ( $value as $item ) {
					if ( ! is_array( $item ) ) {
						$set_error();
					} elseif ( empty( $item['slug'] ) ) {
						$set_error( esc_html__( 'All "Slug" fields are required.', 'redparts-sputnik' ) );
					} elseif ( 0 === preg_match( '/^[-_a-z]+$/', $item['slug'] ) ) {
						$set_error( esc_html__( 'The "Slug" field can only contain the following characters: -_a-z.', 'redparts-sputnik' ) );
					} elseif ( in_array( $item['slug'], $all_slugs, true ) ) {
						$set_error( esc_html__( 'All "Slug" fields must be unique.', 'redparts-sputnik' ) );
					} elseif ( empty( $item['label'] ) ) {
						$set_error( esc_html__( 'All "Label" fields are required.', 'redparts-sputnik' ) );
					} elseif ( empty( $item['placeholder'] ) ) {
						$set_error( esc_html__( 'All "Placeholder" fields are required.', 'redparts-sputnik' ) );
					}

					$all_slugs[] = $item['slug'];
				}
			}

			$return['value'] = $value;

			if ( true === $error ) {
				$field['msg']    = $message;
				$return['error'] = $field;
			}

			return $return;
		}
	}
}
