<?php
/**
 * Block banner widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Banner' ) ) {
	/**
	 * Class Block_Banner
	 */
	class Block_Banner extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_banner',
				'description'                 => esc_html_x( 'Displays a banners.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_banner',
				esc_html_x( 'RedParts: Block Banner', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$banners_raw = isset( $instance['banners'] ) && is_array( $instance['banners'] ) ? array_values( $instance['banners'] ) : array();
			$banners     = array();

			foreach ( $banners_raw as $banner_raw ) {
				$banner_raw = is_array( $banner_raw ) ? $banner_raw : array();

				$banners[] = array(
					'style'    => isset( $banner_raw['style'] ) ? $banner_raw['style'] : '',
					'order'    => isset( $banner_raw['order'] ) ? $banner_raw['order'] : 0,
					'url'      => isset( $banner_raw['url'] ) ? $banner_raw['url'] : '',
					'title'    => isset( $banner_raw['title'] ) ? $banner_raw['title'] : '',
					'subtitle' => isset( $banner_raw['subtitle'] ) ? $banner_raw['subtitle'] : '',
					'button'   => isset( $banner_raw['button'] ) ? $banner_raw['button'] : '',
					'image'    => isset( $banner_raw['image'] ) ? absint( $banner_raw['image'] ) : '',
				);
			}

			uasort(
				$banners,
				function( $a, $b ) {
					$a = intval( $a['order'] );
					$b = intval( $b['order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			?>
			<!-- .th-block-banner -->
			<div class="th-block-banners th-block">
				<div class="th-container">
					<div class="th-block-banners__list">
						<?php foreach ( $banners as $banner ) : ?>
							<a
								href="<?php echo esc_url( $banner['url'] ); ?>"
								class="th-block-banners__item th-block-banners__item--style--<?php echo esc_attr( $banner['style'] ); ?>"
							>
								<?php if ( ! empty( $banner['image'] ) ) : ?>
									<span class="th-block-banners__item-image">
										<?php echo wp_get_attachment_image( $banner['image'], 'full' ); ?>
									</span>
									<span class="th-block-banners__item-image th-block-banners__item-image--blur">
										<?php echo wp_get_attachment_image( $banner['image'], 'full' ); ?>
									</span>
								<?php endif; ?>
								<?php if ( ! empty( $banner['title'] ) ) : ?>
									<span class="th-block-banners__item-title">
										<?php echo wp_kses( $banner['title'], 'redparts_sputnik_block_banner_title' ); ?>
									</span>
								<?php endif; ?>
								<?php if ( ! empty( $banner['subtitle'] ) ) : ?>
									<span class="th-block-banners__item-details">
										<?php echo wp_kses( $banner['subtitle'], 'redparts_sputnik_block_banner_subtitle' ); ?>
									</span>
								<?php endif; ?>
								<?php if ( ! empty( $banner['button'] ) ) : ?>
									<span class="th-block-banners__item-button th-button th-button--style--primary th-button--size--small">
										<?php echo esc_html( $banner['button'] ); ?>
									</span>
								<?php endif; ?>
							</a>
						<?php endforeach; ?>
					</div>
				</div>
			</div>
			<!-- .th-block-banner / end -->
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'        => 'banners',
					'select_key' => 'current_banner',
					'label'      => esc_html__( 'Banners', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'max'        => 2,
					'fields'     => array(
						array(
							'key'     => 'style',
							'label'   => esc_html__( 'Style', 'redparts-sputnik' ),
							'type'    => 'select',
							'options' => array(
								array(
									'key'   => 'one',
									'label' => esc_html__( 'One', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'two',
									'label' => esc_html__( 'Two', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
						array(
							'key'          => 'url',
							'label'        => esc_html__( 'URL', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'subtitle',
							'label'        => esc_html__( 'Subtitle', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'button',
							'label'        => esc_html__( 'Button Text', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'   => 'image',
							'label' => esc_html__( 'Image', 'redparts-sputnik' ),
							'type'  => 'image',
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Banner', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Banner', 'redparts-sputnik' ),
						/* translators: %s: Group number. */
						'title'  => esc_html__( 'Banner %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_media();
		}
	}
}
