<?php
/**
 * Block categories widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Term;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Categories' ) ) {
	/**
	 * Class Block_Categories
	 */
	class Block_Categories extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_categories',
				'description'                 => esc_html_x( "A list of your store's categories.", 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_categories',
				esc_html_x( 'RedParts: Block Categories', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$options = wp_parse_args(
				$instance,
				array(
					'title'      => '',
					'categories' => '',
				)
			);

			$category_slugs       = wp_parse_slug_list( $options['categories'] );
			$number               = absint( ! empty( $options['number'] ) ? $options['number'] : 0 );
			$subcategories_number = absint( ! empty( $options['subcategories_number'] ) ? $options['subcategories_number'] : 0 );
			$hide_empty           = ! empty( $options['hide_empty'] ) && 'yes' === $options['hide_empty'];

			$categories = get_categories(
				array(
					'taxonomy'   => 'product_cat',
					'hide_empty' => $hide_empty,
					'slug'       => $category_slugs,
					'number'     => $number,
					'orderby'    => 'slug__in',
				)
			)

			?>
			<!-- .th-block-categories -->
			<div class="th-block th-block-categories">
				<div class="th-container">
					<div class="th-block-categories__header">
						<div class="th-block-categories__title">
							<?php echo esc_html( $options['title'] ); ?>

							<div class="th-decor th-decor--type--center th-block-categories__title-decor">
								<div class="th-decor__body">
									<div class="th-decor__start"></div>
									<div class="th-decor__end"></div>
									<div class="th-decor__center"></div>
								</div>
							</div>
						</div>
					</div>
				</div>
				<div class="th-block-categories__body">
					<div class="th-decor th-decor--type--bottom th-block-categories__body-decor">
						<div class="th-decor__body">
							<div class="th-decor__start"></div>
							<div class="th-decor__end"></div>
							<div class="th-decor__center"></div>
						</div>
					</div>

					<div class="th-container">
						<div class="th-block-categories__list">
							<?php foreach ( $categories as $category ) : ?>
								<?php
								/** Category object. @var WP_Term $category */

								$image = get_term_meta( $category->term_id, 'thumbnail_id', true );
								$image = wp_get_attachment_image( $image, array( 160, 160 ) );

								$subcategories = get_categories(
									array(
										'taxonomy'   => 'product_cat',
										'hide_empty' => $hide_empty,
										'parent'     => $category->term_id,
										'number'     => $subcategories_number,
									)
								);

								?>
								<div class="th-block-categories__item th-category-card th-category-card--layout--classic">
									<div class="th-category-card__body">
										<div class="th-category-card__content">
											<?php if ( $image ) : ?>
												<div class="th-category-card__image">
													<a href="<?php echo esc_url( get_term_link( $category->term_id, 'product_cat' ) ); ?>">
														<?php echo wp_kses( $image, 'redparts_sputnik_image' ); ?>
													</a>
												</div>
											<?php endif; ?>

											<div class="th-category-card__info">
												<div class="th-category-card__name">
													<a href="<?php echo esc_url( get_term_link( $category->term_id, 'product_cat' ) ); ?>">
														<?php echo esc_html( $category->name ); ?>
													</a>
												</div>

												<?php if ( $subcategories ) : ?>
													<ul class="th-category-card__children">
														<?php foreach ( $subcategories as $subcategory ) : ?>
															<?php $url = get_term_link( $subcategory->term_id, 'product_cat' ); ?>
															<li>
																<a href="<?php echo esc_url( $url ); ?>">
																	<?php echo esc_html( $subcategory->name ); ?>
																</a>
															</li>
														<?php endforeach; ?>
													</ul>
												<?php endif; ?>

												<div class="th-category-card__actions">
													<a
														class="th-category-card__link"
														href="<?php echo esc_url( get_term_link( $category->term_id, 'product_cat' ) ); ?>"
													>
														<?php esc_attr_e( 'Show All', 'redparts-sputnik' ); ?>
													</a>
												</div>
											</div>
										</div>
									</div>
								</div>
							<?php endforeach; ?>
						</div>
					</div>
				</div>
			</div>
			<!-- .th-block-categories / end -->
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'   => 'categories',
					'label' => esc_html__( 'Categories (comma separated slugs)', 'redparts-sputnik' ),
					'type'  => 'text',
				),
				array(
					'key'     => 'number',
					'label'   => esc_html__( 'Number of categories to show', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => 6,
				),
				array(
					'key'     => 'subcategories_number',
					'label'   => esc_html__( 'Number of sub-categories to show', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => 5,
				),
				array(
					'key'     => 'hide_empty',
					'label'   => esc_html__( 'Hide empty categories', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
				),
			);
		}
	}
}
