<?php
/**
 * Block reviews widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Reviews' ) ) {
	/**
	 * Class Block_Reviews
	 */
	class Block_Reviews extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_reviews',
				'description'                 => esc_html_x( 'Displays reviews carousel.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_reviews',
				esc_html_x( 'RedParts: Block Reviews', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$title       = ! empty( $instance['title'] ) ? $instance['title'] : '';
			$subtitle    = ! empty( $instance['subtitle'] ) ? $instance['subtitle'] : '';
			$reviews_raw = isset( $instance['reviews'] ) && is_array( $instance['reviews'] ) ? array_values( $instance['reviews'] ) : array();
			$reviews     = array();

			foreach ( $reviews_raw as $review_raw ) {
				$review_raw = is_array( $review_raw ) ? $review_raw : array();

				$reviews[] = array(
					'text'     => isset( $review_raw['text'] ) ? $review_raw['text'] : '',
					'rating'   => isset( $review_raw['rating'] ) ? absint( $review_raw['rating'] ) : '',
					'name'     => isset( $review_raw['name'] ) ? $review_raw['name'] : '',
					'position' => isset( $review_raw['position'] ) ? $review_raw['position'] : '',
					'avatar'   => isset( $review_raw['avatar'] ) ? absint( $review_raw['avatar'] ) : '',
					'order'    => isset( $review_raw['order'] ) ? $review_raw['order'] : 0,
				);
			}

			uasort(
				$reviews,
				function( $a, $b ) {
					$a = intval( $a['order'] );
					$b = intval( $b['order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			$start_position = array_search( intval( $instance['current_review'] ), array_keys( $reviews ), true );
			$start_position = $start_position && ( $this->is_preview() || $this->is_elementor_editor() ) ? $start_position : 0;

			?>
			<!-- .th-block-reviews -->
			<div
				class="th-block th-block-reviews"
				data-start-position="<?php echo esc_attr( $start_position ); ?>"
			>
				<div class="th-container th-container--max--xl">
					<?php if ( ! empty( $title ) || ! empty( $subtitle ) ) : ?>
						<div class="th-block-reviews__header">
							<?php if ( ! empty( $title ) ) : ?>
								<div class="th-block-reviews__title">
									<?php echo wp_kses( $title, 'redparts_sputnik_block_reviews_title' ); ?>
								</div>
							<?php endif; ?>
							<?php if ( ! empty( $subtitle ) ) : ?>
								<div class="th-block-reviews__subtitle">
									<?php echo wp_kses( $subtitle, 'redparts_sputnik_block_reviews_subtitle' ); ?>
								</div>
							<?php endif; ?>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $reviews ) ) : ?>
						<div class="th-block-reviews__list">
							<div class="owl-carousel">
								<?php foreach ( $reviews as $review ) : ?>
									<div class="th-block-reviews__item">
										<?php if ( wp_attachment_is_image( $review['avatar'] ) ) : ?>
											<div class="th-block-reviews__item-avatar">
												<div class="th-block-reviews__item-avatar-shadow"></div>
												<div class="th-block-reviews__item-avatar-image">
													<?php echo wp_get_attachment_image( $review['avatar'], array( 80, 80 ) ); ?>
												</div>
											</div>
										<?php endif; ?>
										<div class="th-block-reviews__item-author">
											<div class="th-block-reviews__item-author-name">
												<?php echo wp_kses( $review['name'], 'redparts_sputnik_text' ); ?>
											</div>
											<div class="th-block-reviews__item-author-position">
												<?php echo wp_kses( $review['position'], 'redparts_sputnik_text' ); ?>
											</div>
										</div>
										<div class="th-block-reviews__item-text">
											<?php echo wp_kses( $review['text'], 'redparts_sputnik_text' ); ?>
										</div>
										<?php if ( function_exists( 'wc_get_star_rating_html' ) ) : ?>
											<div class="th-block-reviews__item-rating">
												<?php
												/* translators: %s: rating */
												$label = sprintf( esc_html__( 'Rated %s out of 5', 'redparts-sputnik' ), $review['rating'] );
												?>
												<div
													aria-label="<?php echo esc_attr( $label ); ?>"
													class="th-product-card__rating-stars star-rating"
													role="img"
												>
													<?php echo wp_kses( wc_get_star_rating_html( $review['rating'] ), 'redparts_sputnik_star_rating' ); ?>
												</div>
											</div>
										<?php endif; ?>
									</div>
								<?php endforeach; ?>
							</div>
						</div>
					<?php endif; ?>
				</div>
			</div>
			<!-- .th-block-reviews / end -->
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'          => 'subtitle',
					'label'        => esc_html__( 'Subtitle', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => true,
				),
				array(
					'key'        => 'reviews',
					'select_key' => 'current_review',
					'label'      => esc_html__( 'Reviews', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'fields'     => array(
						array(
							'key'          => 'text',
							'label'        => esc_html__( 'Text', 'redparts-sputnik' ),
							'type'         => 'textarea',
							'translatable' => true,
						),
						array(
							'key'   => 'rating',
							'label' => esc_html__( 'Rating', 'redparts-sputnik' ),
							'type'  => 'number',
						),
						array(
							'key'          => 'name',
							'label'        => esc_html__( 'Name', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'position',
							'label'        => esc_html__( 'Position', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'   => 'avatar',
							'label' => esc_html__( 'Avatar', 'redparts-sputnik' ),
							'type'  => 'image',
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Review', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Review', 'redparts-sputnik' ),
						/* translators: %s: Group number. */
						'title'  => esc_html__( 'Review %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-reviews' );
		}
	}
}
