( function( $ ) {
	'use strict';

	/* global redPartsSputnikVars */

	function initControls() {
		initTabsControls();
		initImagePickerControls();

		if ( $.fn.hasOwnProperty( 'wpColorPicker' ) ) {
			$( '.redparts-color-picker' ).wpColorPicker();
		}
	}

	function initTabsControls() {
		$( '.redparts-tabs' ).each( function( index, element ) {
			element = $( element );

			if ( ! element.data( 'redparts-tabs-instance' ) ) {
				element.data( 'redparts-tabs-instance', new TabsControl( element ) );
			}
		} );
	}

	function initImagePickerControls() {
		$( '.redparts-image-picker' ).each( function( index, element ) {
			element = $( element );

			if ( ! element.data( 'redparts-image-picker-instance' ) ) {
				element.data( 'redparts-image-picker-instance', new ImagePickerControl( element ) );
			}
		} );
	}

	function TabsControl( element ) {
		this.element = $( element );
		this.addTab = this.addTab.bind( this );
		this.initPanel = this.initPanel.bind( this );
		this.onChangePanel = this.onChangePanel.bind( this );

		this.addButton = this.element.find( '> .redparts-tabs__add' );
		this.tabsList = this.element.find( '> .redparts-tabs__list' );
		this.panelTemplate = _.template( this.element.find( '> .redparts-tabs__panel-template' ).html() );
		this.panelTitle = this.element.data( 'panel-title' );

		this.addButton.on( 'click', this.addTab );
		this.tabsList.on( 'change', this.onChangePanel );

		this.panels = _.map( this.getPanels().toArray(), this.initPanel );
	}
	TabsControl.prototype.addTab = function( event ) {
		if ( ! event.cancelable ) {
			return;
		}

		event.preventDefault();

		const index = this.getNextIndex();

		const panelElement = $( this.panelTemplate( { index: index } ) )[ 0 ];
		const panel = this.initPanel( panelElement );

		this.panels.push( panel );
		this.element.append( panelElement );
		// noinspection JSUnresolvedFunction
		this.tabsList.append( '<option value="' + index + '">' + wp.i18n.sprintf( this.panelTitle, index + 1 ) + '</option>' );

		const panels = this.getPanels();

		panels.filter( ':not([data-index="' + index + '"])' ).removeClass( 'redparts-tabs__panel--active' );
		panels.filter( '[data-index="' + index + '"]' ).addClass( 'redparts-tabs__panel--active' );

		this.tabsList.prop( 'value', index );
		this.checkMaxPanels();

		initControls();

		this.tabsList.trigger( 'change' );
	};
	TabsControl.prototype.removeTab = function( panel ) {
		let index = panel.index;

		this.panels.splice( this.panels.indexOf( panel ), 1 );

		const panels = this.getPanels();

		this.tabsList.find( 'option[value="' + index + '"]' ).remove();

		panel.element.remove();

		const option = this.tabsList.find( 'option' ).first();

		if ( option.length ) {
			index = option.val();

			this.tabsList.prop( 'value', index );

			panels.filter( '[data-index="' + index + '"]' ).addClass( 'redparts-tabs__panel--active' );
		}

		this.checkMaxPanels();

		this.tabsList.trigger( 'change' );
	};
	TabsControl.prototype.getNextIndex = function() {
		return _.reduce( this.tabsList.find( 'option' ).toArray(), function( index, option ) {
			const optionIndex = parseFloat( option.value );

			return optionIndex >= index ? optionIndex + 1 : index;
		}, 0 );
	};
	TabsControl.prototype.getPanels = function() {
		return this.element.find( '> .redparts-tabs__panel' );
	};
	TabsControl.prototype.checkMaxPanels = function() {
		const max = this.element.data( 'max' ) ? parseFloat( this.element.data( 'max' ) ) : 0;

		this.element.toggleClass( 'redparts-tabs--max', 0 < max && max <= this.getPanels().length );
	};
	TabsControl.prototype.initPanel = function( panelElement ) {
		const self = this;
		const panel = new TabsPanelControl( panelElement );

		panel.removeButton.on( 'click', function( event ) {
			if ( ! event.cancelable ) {
				return;
			}

			event.preventDefault();

			self.removeTab( panel );
		} );

		return panel;
	};
	TabsControl.prototype.onChangePanel = function() {
		const index = this.tabsList.val();
		const panels = this.getPanels();

		panels.filter( ':not([data-index="' + index + '"])' ).removeClass( 'redparts-tabs__panel--active' );
		panels.filter( '[data-index="' + index + '"]' ).addClass( 'redparts-tabs__panel--active' );
	};

	function TabsPanelControl( element ) {
		this.element = $( element );
		this.index = this.element.data( 'index' );
		this.removeButton = this.element.find( '> .redparts-tabs__panel-footer > .redparts-tabs__remove' );
	}

	function ImagePickerControl( element ) {
		this.element = $( element );
		this.selectImage = this.selectImage.bind( this );
		this.removeImage = this.removeImage.bind( this );
		this.selectButton = this.element.find( '.redparts-image-picker__button, .redparts-image-picker__replace' );
		this.removeButton = this.element.find( '.redparts-image-picker__remove' );
		this.container = this.element.find( '.redparts-image-picker__container' );
		this.input = this.element.find( 'input' );

		this.selectButton.on( 'click', this.selectImage );
		this.removeButton.on( 'click', this.removeImage );
	}
	ImagePickerControl.prototype.selectImage = function( event ) {
		if ( ! event.cancelable ) {
			return;
		}

		event.preventDefault();

		const self = this;
		const frame = wp.media( {
			title: redPartsSputnikVars.selectImage,
			button: {
				text: redPartsSputnikVars.selectImage,
			},
			multiple: false,
		} );

		// noinspection JSCheckFunctionSignatures
		frame.on( 'select', function() {
			// noinspection JSCheckFunctionSignatures
			self.setImage( frame.state().get( 'selection' ).first().toJSON() );
		} );

		// noinspection JSCheckFunctionSignatures
		frame.on( 'close', function() {
			frame.detach();
		} );

		frame.open();
	};
	ImagePickerControl.prototype.setImage = function( attachment ) {
		this.element.addClass( 'redparts-image-picker--has-image' );
		this.container.html( '<img src="' + attachment.url + '" alt="">' );
		this.input.val( attachment.id ).trigger( 'change' );
	};
	ImagePickerControl.prototype.removeImage = function( event ) {
		if ( ! event.cancelable ) {
			return;
		}

		event.preventDefault();

		this.element.removeClass( 'redparts-image-picker--has-image' );
		this.container.html( '' );
		this.input.val( '' ).trigger( 'change' );
	};

	$( document ).on( 'widget-added widget-updated', function() {
		initControls();
	} );

	$( function() {
		initControls();

		if ( window.elementor && elementor.hooks ) {
			[
				'redparts_sputnik_block_about_us',
				'redparts_sputnik_block_attributes_finder',
				'redparts_sputnik_block_banner',
				'redparts_sputnik_block_brands',
				'redparts_sputnik_block_categories',
				'redparts_sputnik_block_features',
				'redparts_sputnik_block_finder',
				'redparts_sputnik_block_posts_carousel',
				'redparts_sputnik_block_products_carousel',
				'redparts_sputnik_block_products_columns',
				'redparts_sputnik_block_reviews',
				'redparts_sputnik_block_sale',
				'redparts_sputnik_block_slideshow',
				'redparts_sputnik_block_space',
				'redparts_sputnik_block_teammates',
				'redparts_sputnik_block_zone',
				'redparts_sputnik_filter_vehicle',
				'redparts_sputnik_filters',
				'redparts_sputnik_footer_contacts',
				'redparts_sputnik_footer_links',
				'redparts_sputnik_footer_newsletter',
				'redparts_sputnik_recent_comments',
				'redparts_sputnik_recent_posts',
			].forEach( function( widget ) {
				elementor.hooks.addAction( 'panel/widgets/wp-widget-' + widget + '/controls/wp_widget/loaded', function() {
					initControls();
				} );
			} );
		}
	} );
}( jQuery ) );
