<?php
/**
 * RedParts sputnik attributes finder.
 *
 * @package RedParts\Sputnik
 * @since 1.7.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Attributes_Finder' ) ) {
	/**
	 * Class Attributes_Finder.
	 *
	 * @since 1.7.0
	 */
	class Attributes_Finder extends Singleton {
		const MAX_PROCESSED_FIELDS = 8;

		/**
		 * Initialization.
		 *
		 * @since 1.7.0
		 */
		public function init() {
			if ( ! wp_doing_ajax() ) {
				return;
			}

			add_action( 'wp_ajax_redparts_sputnik_attributes_finder', array( $this, 'ajax_get_options' ) );
			add_action( 'wp_ajax_nopriv_redparts_sputnik_attributes_finder', array( $this, 'ajax_get_options' ) );
		}

		/**
		 * Handles "get options" AJAX request.
		 *
		 * @since 1.7.0
		 */
		public function ajax_get_options() {
			$max_fields_processed = self::MAX_PROCESSED_FIELDS;

			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['nonce'] ) ), 'redparts_sputnik_attributes_finder' )
			) {
				wp_die( esc_html__( 'Action failed. Please refresh the page and retry.', 'redparts-sputnik' ) );
			}

			WPML::switch_ajax_language();

			if (
				! isset( $_POST['data']['fields'] ) ||
				! isset( $_POST['data']['values'] )
			) {
				wp_die( -1 );
			}

			$fields = array_map( 'sanitize_key', wp_unslash( $_POST['data']['fields'] ) );
			$values = array_map( 'sanitize_key', wp_unslash( $_POST['data']['values'] ) );
			$strict = false;

			if ( isset( $_POST['data']['strict'] ) ) {
				$strict = 1 === absint( wp_unslash( $_POST['data']['strict'] ) );
			}

			if ( 0 === count( $fields ) || $max_fields_processed < count( $fields ) || count( $values ) >= count( $fields ) ) {
				wp_die( -1 );
			}

			if ( false === $strict ) {
				$fields = array_slice( $fields, 0, count( $values ) + 1 );
			}

			foreach ( $fields as $field_idx => $field ) {
				if ( 0 !== strpos( $field, 'filter_' ) ) {
					wp_die( -1 );
				}

				$attribute = wc_sanitize_taxonomy_name( str_replace( 'filter_', '', $field ) );
				$taxonomy  = wc_attribute_taxonomy_name( $attribute );

				if ( ! taxonomy_exists( $taxonomy ) || ! wc_attribute_taxonomy_id_by_name( $attribute ) ) {
					wp_die( -1 );
				}

				$fields[ $field_idx ] = $taxonomy;
			}

			$result = array_map(
				function( $item ) {
					return array(
						'title' => $item['name'],
						'value' => $item['slug'],
					);
				},
				$this->get_options( $fields, $values )
			);

			array_multisort( array_column( $result, 'title' ), SORT_ASC, $result );

			wp_send_json_success( $result );
		}

		/**
		 * Returns options for fields of attribute finder.
		 *
		 * @since 1.7.0
		 *
		 * @param array $fields Array of product attribute slugs.
		 * @param array $values Array of product attributes values.
		 *
		 * @return array|object|null
		 */
		public function get_options( array $fields, array $values = array() ) {
			global $wpdb;

			$fields = array_values( $fields );
			$values = array_values( $values );

			$select_statements   = array();
			$select_placeholders = array();
			$join_statements     = array();
			$join_placeholders   = array();
			$where_statements    = array();
			$where_placeholders  = array();
			$order_statements    = array();

			$where_statements[] = "p.post_type = 'product'";
			$where_statements[] = "p.post_status = 'publish'";

			foreach ( $fields as $idx => $field ) {
				$is_target_field = count( $values ) === $idx;

				$tr = 'tr' . $idx;
				$tt = 'tt' . $idx;
				$t  = 't' . $idx;

				$join_statements[] = "INNER JOIN $wpdb->term_relationships AS $tr ON p.ID = $tr.object_id";
				$join_statements[] = "INNER JOIN $wpdb->term_taxonomy AS $tt ON $tt.term_taxonomy_id = $tr.term_taxonomy_id";
				$join_statements[] = "INNER JOIN $wpdb->terms AS $t ON $t.term_id = $tt.term_id";

				if ( $is_target_field ) {
					$select_statements[] = "$t.*";

					$where_statements[]   = "($tt.taxonomy = %s)";
					$where_placeholders[] = $field;

					$order_statements[] = "$t.name";
				} elseif ( $idx >= count( $values ) ) {
					$where_statements[]   = "($tt.taxonomy = %s)";
					$where_placeholders[] = $field;
				} else {
					$where_statements[]   = "($tt.taxonomy = %s AND $t.slug = %s)";
					$where_placeholders[] = $field;
					$where_placeholders[] = $values[ $idx ];
				}
			}

			$select_statements = implode( ", \n", $select_statements );
			$join_statements   = implode( "\n", $join_statements );
			$where_statements  = implode( " AND\n", $where_statements );
			$order_statements  = implode( ', ', $order_statements );

			$placeholders = array_merge(
				$select_placeholders,
				$join_placeholders,
				$where_placeholders
			);

			$sql = "
				SELECT DISTINCT
				$select_statements
				FROM $wpdb->posts AS p
				$join_statements
				WHERE
				$where_statements
				ORDER BY $order_statements ASC
			";

			// phpcs:disable WordPress.DB.DirectDatabaseQuery
			// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
			$query = $wpdb->prepare( $sql, ...$placeholders );

			return $wpdb->get_results( $query, ARRAY_A );
			// phpcs:enable
		}
	}
}
