<?php
/**
 * RedParts featured attributes.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WC_Product;
use WC_Product_Attribute;
use Exception;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Featured_Attributes' ) ) {
	/**
	 * Class Featured_Attributes
	 */
	class Featured_Attributes extends Singleton {
		const META_KEY = '_redparts_featured_attributes';

		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'woocommerce_after_product_attribute_settings', array( $this, 'product_attribute_settings_template' ), 10, 2 );
			add_action( 'wp_ajax_woocommerce_save_attributes', array( $this, 'ajax_save_featured_attributes' ), 5 );
		}

		/**
		 * Renders additional attribute settings.
		 *
		 * @param WC_Product_Attribute $attribute Attribute object.
		 * @param int                  $index Attribute index.
		 */
		public function product_attribute_settings_template( WC_Product_Attribute $attribute, int $index ) {
			?>
			<tr>
				<td>
					<label>
						<input
							type="checkbox"
							class="checkbox"
							name="redparts_attribute_featured[<?php echo esc_attr( $index ); ?>]"
							value="1"
							<?php checked( $this->is_featured_attribute( $attribute ) ); ?>
						/>
						<?php esc_html_e( 'Featured', 'redparts-sputnik' ); ?>
					</label>
				</td>
			</tr>
			<?php
		}

		/**
		 * Returns true if attribute is featured.
		 *
		 * @param WC_Product_Attribute $attribute Attribute.
		 * @return bool
		 */
		public function is_featured_attribute( WC_Product_Attribute $attribute ): bool {
			// Add or edit an attribute.
			if ( wp_doing_ajax() ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Missing
				$post_id = isset( $_POST['post_id'] ) ? absint( wp_unslash( $_POST['post_id'] ) ) : 0;
			} else {
				global $post;

				$post_id = $post->ID;
			}

			$featured_attributes = (array) get_post_meta( WPML::get_original_product_id( absint( $post_id ) ), self::META_KEY, true );

			return in_array( $attribute->get_name(), $featured_attributes, true );
		}

		/**
		 * Saves attribute settings.
		 */
		public function ajax_save_featured_attributes() {
			check_ajax_referer( 'save-attributes', 'security' );

			if ( ! current_user_can( 'edit_products' ) || ! isset( $_POST['data'], $_POST['post_id'] ) ) {
				wp_die( -1 );
			}

			try {
				// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				parse_str( wp_unslash( $_POST['data'] ), $data );

				$product_id          = absint( wp_unslash( $_POST['post_id'] ) );
				$featured_attributes = array();

				if ( ! empty( $data['attribute_names'] ) && ! empty( $data['redparts_attribute_featured'] ) ) {
					$attribute_names = $data['attribute_names'];

					foreach ( array_keys( $data['redparts_attribute_featured'] ) as $index ) {
						if ( empty( $attribute_names[ $index ] ) ) {
							continue;
						}

						$featured_attributes[] = wc_clean( $attribute_names[ $index ] );
					}
				}

				$featured_attributes = apply_filters( 'redparts_sputnik_ajax_save_featured_attributes', $featured_attributes, $product_id );

				update_post_meta( $product_id, self::META_KEY, $featured_attributes );
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) );
			}
		}

		/**
		 * Returns slugs of the featured attributes.
		 *
		 * @param WC_Product $product Product.
		 * @return WC_Product_Attribute[]
		 */
		public function get_featured_attributes( WC_Product $product ): array {
			/**
			 * Array of product attributes.
			 *
			 * @var WC_Product_Attribute[] $attributes
			 */
			$attributes = $product->get_attributes();
			$featured   = (array) get_post_meta( WPML::get_original_product_id( $product->get_id() ), self::META_KEY, true );
			$result     = array();

			if ( function_exists( 'wc_attribute_taxonomy_name' ) ) {
				$global_featured = (array) Settings::instance()->get( 'global_featured_attributes', array() );
				$global_featured = array_map( 'wc_attribute_taxonomy_name', $global_featured );

				$featured = array_merge( $featured, $global_featured );
			}

			if ( is_array( $featured ) ) {
				$result = array_filter(
					$attributes,
					function( $attribute ) use ( $featured ) {
						return ! is_string( $attribute ) && in_array( $attribute->get_name(), $featured, true );
					}
				);
			}

			return apply_filters( 'redparts_sputnik_get_featured_attributes', $result, $product );
		}

		/**
		 * Checks if a product has featured attributes.
		 *
		 * @since 1.6.0
		 *
		 * @param WC_Product $product Product.
		 * @return bool
		 */
		public function has_featured_attributes( WC_Product $product ): bool {
			return ! empty( $this->get_featured_attributes( $product ) );
		}

		/**
		 * Outputs featured attributes.
		 *
		 * @param string $classes CSS classes.
		 */
		public function the_featured_attributes( string $classes = '' ) {
			global $product;

			if ( $classes ) {
				$classes = preg_split( '#\s+#', $classes );
			} else {
				$classes = array();
			}

			$classes[] = 'th-features';

			$attributes         = $this->get_featured_attributes( $product );
			$product_attributes = array();

			foreach ( $attributes as $attribute ) {
				$values = array();

				if ( $attribute->is_taxonomy() ) {
					$attribute_values = wc_get_product_terms( $product->get_id(), $attribute->get_name(), array( 'fields' => 'all' ) );

					foreach ( $attribute_values as $attribute_value ) {
						$values[] = $attribute_value->name;
					}
				} else {
					foreach ( $attribute->get_options() as $value ) {
						$values[] = $value;
					}
				}

				$product_attributes[] = array(
					'key'    => sanitize_title_with_dashes( $attribute->get_name() ),
					'label'  => wc_attribute_label( $attribute->get_name() ),
					'values' => wptexturize( implode( ', ', $values ) ),
				);
			}

			if ( $product_attributes ) :
				?>
				<ul class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>">
					<?php foreach ( $product_attributes as $attribute ) : ?>
						<li class="th-features__item th-features__item--key--<?php echo esc_attr( $attribute['key'] ); ?>">
							<span class="th-features__item-label"><?php echo esc_html( $attribute['label'] ); ?></span>:
							<span class="th-features__item-values"><?php echo esc_html( $attribute['values'] ); ?></span>
						</li>
					<?php endforeach; ?>
				</ul>
				<?php
			endif;
		}
	}
}
