<?php
/**
 * RedParts sputnik vehicles.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 * @noinspection SqlResolve
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WP_Term;
use WC_Product_Attribute;
use WC_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Vehicles' ) ) {
	/**
	 * Class Vehicles.
	 *
	 * @package RedParts\Sputnik
	 */
	class Vehicles extends Singleton {
		/**
		 * Form fields.
		 *
		 * @var array
		 */
		protected static $fields = array();

		/**
		 * Form fields.
		 *
		 * @var array
		 */
		protected static $admin_fields = array();

		/**
		 * Initialization.
		 */
		public function init() {
			// The priority is 100 so that Redux has time to initialize.
			add_action( 'after_setup_theme', array( $this, 'deferred_init' ), 100 );
		}

		/**
		 * Deferred initialization.
		 */
		public function deferred_init() {
			if ( 'yes' !== Settings::instance()->get( 'autoparts_features' ) ) {
				return;
			}

			add_action( 'delete_term', array( $this, 'on_delete_term' ) );

			self::$fields       = array_values( Settings::instance()->get( 'vehicle_fields' ) );
			self::$admin_fields = array();

			foreach ( self::$fields as $field ) {
				if ( 'year' === $field['type'] ) {
					self::$admin_fields[] = array(
						'name'  => 'redparts_' . $field['slug'] . '_since',
						// translators: %s field name, for example: Year.
						'title' => sprintf( esc_html__( 'Vehicle %s (since)', 'redparts-sputnik' ), $field['label'] ),
					);
					self::$admin_fields[] = array(
						'name'  => 'redparts_' . $field['slug'] . '_until',
						// translators: %s field name, for example: Year.
						'title' => sprintf( esc_html__( 'Vehicle %s (until)', 'redparts-sputnik' ), $field['label'] ),
					);
				} else {
					self::$admin_fields[] = array(
						'name'  => 'redparts_' . $field['slug'],
						// translators: %s field name, for example: Model.
						'title' => sprintf( esc_html__( 'Vehicle %s', 'redparts-sputnik' ), $field['label'] ),
					);
				}
			}

			if ( 'yes' === Settings::instance()->get( 'search_vehicles_by_vin' ) ) {
				self::$admin_fields[] = array(
					'name'  => 'redparts_vin',
					'title' => esc_html__( 'Vehicle VIN', 'redparts-sputnik' ),
				);
			}

			// Reindex vehicle fields table after import.
			add_action( 'pt-ocdi/after_import', array( $this, 'reindex_vehicle_fields_table' ), 20 );
			// Clear options cache after import.
			add_action( 'pt-ocdi/after_import', array( $this, 'clear_get_options_cache' ), 30 );

			$attribute_slug = $this->get_attribute_slug();

			if ( $attribute_slug ) {
				add_action( $attribute_slug . '_add_form_fields', array( $this, 'add_form_fields' ) );
				add_action( $attribute_slug . '_edit_form_fields', array( $this, 'edit_form_fields' ) );
				add_action( 'create_' . $attribute_slug, array( $this, 'save_form' ) );
				add_action( 'edit_' . $attribute_slug, array( $this, 'save_form' ) );
				add_action( 'init', array( $this, 'replace_vehicle_instance_id_with_vehicle_slug' ) );

				add_action( 'saved_' . $attribute_slug, array( $this, 'clear_get_options_cache' ) );
				add_action( 'delete_' . $attribute_slug, array( $this, 'clear_get_options_cache' ) );
			}

			add_shortcode( 'redparts_sputnik_vehicle_select', array( $this, 'vehicle_select_shortcode' ) );
			add_shortcode( 'redparts_sputnik_vehicle_form', array( $this, 'vehicle_form_shortcode' ) );

			if ( 'yes' === Settings::instance()->get( 'search_vehicles_by_vin' ) ) {
				add_shortcode( 'redparts_sputnik_vehicle_vin', array( $this, 'vehicle_vin_shortcode' ) );
			}

			if ( ! wp_doing_ajax() ) {
				return;
			}

			add_action( 'wp_ajax_redparts_sputnik_vehicle_select_load_data', array( $this, 'ajax_vehicle_select_load_data' ) );
			add_action( 'wp_ajax_nopriv_redparts_sputnik_vehicle_select_load_data', array( $this, 'ajax_vehicle_select_load_data' ) );

			if ( 'yes' === Settings::instance()->get( 'search_vehicles_by_vin' ) ) {
				add_action( 'wp_ajax_redparts_sputnik_vehicle_vin_load_data', array( $this, 'ajax_vehicle_vin_load_data' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_vehicle_vin_load_data', array( $this, 'ajax_vehicle_vin_load_data' ) );
			}

			add_action( 'wp_ajax_redparts_sputnik_vehicles_export', array( $this, 'ajax_export' ) );
			add_action( 'wp_ajax_redparts_sputnik_vehicles_import', array( $this, 'ajax_import' ) );
			add_action( 'wp_ajax_redparts_sputnik_vehicles_clear', array( $this, 'ajax_clear' ) );
		}

		/**
		 * Returns the vehicle attribute slug.
		 *
		 * @return string|null
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public function get_attribute_slug() {
			if ( ! function_exists( 'wc_attribute_taxonomy_name' ) ) {
				return null;
			}

			$attribute_name = Settings::instance()->get( 'vehicle_attribute' );

			if ( empty( $attribute_name ) ) {
				return null;
			}

			return sanitize_key( wc_attribute_taxonomy_name( $attribute_name ) );
		}

		/**
		 * Replaces vehicle instance ID with vehicle slug in the $_GET global variable.
		 *
		 * @since 1.7.0
		 */
		public function replace_vehicle_instance_id_with_vehicle_slug() {
			$attribute_filter_name = $this->get_attribute_filter_name();

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			if ( ! isset( $_GET[ $attribute_filter_name ] ) ) {
				return;
			}

			$attribute_filter_value = sanitize_text_field( wp_unslash( $_GET[ $attribute_filter_name ] ) );

			if ( ! preg_match( '#^[0-9]+:#', $attribute_filter_value ) ) {
				return;
			}

			$_GET[ 'redparts_' . $attribute_filter_name ] = $_GET[ $attribute_filter_name ];
			// phpcs:enable

			$vehicle_instance_id = self::decode_vehicle_instance_id( $attribute_filter_value );

			if ( ! $vehicle_instance_id ) {
				return;
			}

			if ( 0 === $vehicle_instance_id['vehicle_id'] ) {
				$vehicle = $this->get_vehicle_by_field_values( $vehicle_instance_id['field_values'] );

				if ( ! $vehicle ) {
					return;
				}

				$slugs = implode( ',', $vehicle['slug'] );

				if ( empty( $slugs ) ) {
					return;
				}
			} else {
				$slugs = get_term( $vehicle_instance_id['vehicle_id'], $this->get_attribute_slug() )->slug;
			}

			$_GET[ $attribute_filter_name ]     = addslashes( $slugs );
			$_REQUEST[ $attribute_filter_name ] = addslashes( $slugs );
		}

		/**
		 * Returns the filter name of the vehicle attribute.
		 *
		 * @return string
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public function get_attribute_filter_name() {
			$attribute_name = Settings::instance()->get( 'vehicle_attribute' );

			if ( empty( $attribute_name ) ) {
				return '';
			}

			return 'filter_' . $attribute_name;
		}

		/**
		 * Returns the filter value of the vehicle attribute.
		 *
		 * @since 1.7.0
		 *
		 * @param array $vehicle Vehicle.
		 *
		 * @return string
		 */
		public function get_attribute_filter_value( array $vehicle ): string {
			return $vehicle['instance_id'];
		}

		/**
		 * Returns the [ALL] term slug.
		 *
		 * @return string
		 */
		public function get_all_term_slug(): string {
			$attribute_slug = $this->get_attribute_slug();

			if ( ! $attribute_slug ) {
				return '__all__';
			}

			$term = get_term_by( 'slug', '__all__', $attribute_slug );

			if ( ! is_object( $term ) || is_wp_error( $term ) ) {
				return '__all__';
			}

			return $term->slug;
		}

		/**
		 * Adds an additional fields to the vehicle add form.
		 */
		public function add_form_fields() {
			foreach ( self::$admin_fields as $field ) {
				?>
				<div class="form-field term-<?php echo esc_attr( $field['name'] ); ?>-wrap">
					<label for="tag-<?php echo esc_attr( $field['name'] ); ?>">
						<?php echo esc_html( $field['title'] ); ?>
					</label>
					<input
						name="<?php echo esc_attr( $field['name'] ); ?>"
						id="tag-<?php echo esc_attr( $field['name'] ); ?>"
						type="text"
						value=""
					>
				</div>
				<?php
			}
		}

		/**
		 * Adds an additional fields to the vehicle edit form.
		 *
		 * @param WP_Term $term WordPress term object.
		 */
		public function edit_form_fields( WP_Term $term ) {
			foreach ( self::$admin_fields as $field ) {
				$field_value = get_term_meta( $term->term_id, $field['name'], true );
				?>
				<tr class="form-field term-<?php echo esc_attr( $field['name'] ); ?>-wrap">
					<th scope="row">
						<label for="<?php echo esc_attr( $field['name'] ); ?>">
							<?php echo esc_html( $field['title'] ); ?>
						</label>
					</th>
					<td>
						<input
							name="<?php echo esc_attr( $field['name'] ); ?>"
							id="<?php echo esc_attr( $field['name'] ); ?>"
							type="text"
							value="<?php echo esc_attr( $field_value ); ?>"
						>
					</td>
				</tr>
				<?php
			}
		}

		/**
		 * Saves vehicle meta fields.
		 *
		 * @param integer $term_id Term ID.
		 */
		public function save_form( int $term_id ) {
			foreach ( self::$admin_fields as $field ) {
				if ( ! isset( $_POST[ $field['name'] ] ) ) {
					return;
				}
			}

			if ( ! current_user_can( 'edit_term', $term_id ) ) {
				return;
			}
			if ( ! isset( $_POST['_wpnonce'] ) && ! isset( $_POST['_wpnonce_add-tag'] ) ) {
				return;
			}

			// Sanitization here would be redundant.
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if (
				! wp_verify_nonce( wp_unslash( $_POST['_wpnonce_add-tag'] ), 'add-tag' )
				&& ! wp_verify_nonce( wp_unslash( $_POST['_wpnonce'] ), 'update-tag_' . $term_id )
			) {
				return;
			}
			// phpcs:enable

			foreach ( self::$admin_fields as $field ) {
				$field_value = sanitize_text_field( wp_unslash( $_POST[ $field['name'] ] ) );

				update_term_meta( $term_id, $field['name'], $field_value );

				$this->save_vehicle_field( $term_id, $field['name'], $field_value );
			}

			$this->clear_get_options_cache();
		}

		/**
		 * Exports vehicles.
		 */
		public function ajax_export() {
			if ( ! $this->get_attribute_slug() ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'You must first select the vehicle attribute.', 'redparts-sputnik' ),
					)
				);
			}

			WPML::switch_ajax_language();

			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce(
					sanitize_key( wp_unslash( $_POST['nonce'] ) ),
					'redparts_sputnik_vehicles_export'
				)
			) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'Action failed. Please refresh the page and retry.', 'redparts-sputnik' ),
					)
				);
			}

			if ( ! current_user_can( 'manage_product_terms' ) ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'You do not have permission to export vehicles.', 'redparts-sputnik' ),
					)
				);
			}

			$limit         = 100;
			$offset        = 0;
			$compatibility = false;

			if ( isset( $_POST['data']['limit'] ) ) {
				$limit = absint( wp_unslash( $_POST['data']['limit'] ) );
			}
			if ( isset( $_POST['data']['offset'] ) ) {
				$offset = absint( wp_unslash( $_POST['data']['offset'] ) );
			}
			if ( isset( $_POST['data']['compatibility'] ) ) {
				$compatibility = 1 === absint( wp_unslash( $_POST['data']['compatibility'] ) );
			}

			$include = array();
			$limit   = min( 500, $limit );

			if ( $compatibility ) {
				$include[] = 'compatibility';
			}

			$result = $this->export(
				array(
					'limit'   => $limit,
					'offset'  => $offset,
					'include' => $include,
				)
			);

			wp_send_json_success(
				$result
			);
		}

		/**
		 * Export vehicles.
		 *
		 * @noinspection PhpMissingReturnTypeInspection
		 * @noinspection PhpMissingParamTypeInspection
		 *
		 * @param array $args {
		 *     Optional. Array of arguments.
		 *
		 *     @type int      $limit   The maximum number of vehicles for export.
		 *     @type int      $offset  Offset from the beginning of the list.
		 *     @type string[] $include Array of entities to include in export. May include: 'compatibility'.
		 * }
		 *
		 * @return array|null
		 */
		public function export( $args = array() ) {
			$args = wp_parse_args(
				$args,
				array(
					'limit'   => 100,
					'offset'  => 0,
					'include' => array(),
				)
			);

			$attribute_slug = $this->get_attribute_slug();

			// Header.
			$header = array();

			if ( in_array( 'compatibility', $args['include'], true ) ) {
				$header[] = 'product_id';
				$header[] = 'product_sku';
			}

			foreach ( self::$fields as $field ) {
				if ( 'year' === $field['type'] ) {
					$header[] = 'vehicle_' . $field['slug'] . '_since';
					$header[] = 'vehicle_' . $field['slug'] . '_until';
				} else {
					$header[] = 'vehicle_' . $field['slug'];
				}
			}

			// Rows.
			$query = function ( $count = false ) use ( $args, $attribute_slug ) {
				global $wpdb;

				$wc_product_meta_lookup = $wpdb->prefix . 'wc_product_meta_lookup';

				$placeholders = array(
					$attribute_slug,
				);

				$sql_select    = array( 't.term_id AS vehicle_id' );
				$sql_left_join = array( "LEFT JOIN $wpdb->terms AS t ON tt.term_id = t.term_id" );
				$sql_order_by  = array( 't.term_id' );

				if ( in_array( 'compatibility', $args['include'], true ) ) {
					$sql_select[] = 'COALESCE(pml.product_id, "") AS product_id';
					$sql_select[] = 'COALESCE(pml.sku, "") AS product_sku';

					$sql_left_join[] = "LEFT JOIN $wpdb->term_relationships AS tr ON tt.term_taxonomy_id = tr.term_taxonomy_id";
					$sql_left_join[] = "LEFT JOIN $wc_product_meta_lookup AS pml ON tr.object_id = pml.product_id";

					$sql_order_by[] = 'pml.product_id';
				}

				if ( $count ) {
					$sql_select = array( 'COUNT(*)' );
					$sql_limit  = '';
				} else {
					$sql_limit = 'LIMIT %d, %d';

					$placeholders[] = $args['offset'];
					$placeholders[] = $args['limit'];
				}

				$sql_select    = implode( ', ', $sql_select );
				$sql_left_join = implode( "\n", $sql_left_join );
				$sql_order_by  = implode( ', ', $sql_order_by );

				$sql = "
				SELECT $sql_select
				FROM $wpdb->term_taxonomy AS tt
				$sql_left_join
				WHERE tt.taxonomy = %s AND t.slug != '__all__'
				ORDER BY $sql_order_by
				$sql_limit
				";

				// phpcs:disable WordPress.DB.DirectDatabaseQuery
				// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
				$prepared = $wpdb->prepare( $sql, ...$placeholders );

				if ( $count ) {
					return absint( $wpdb->get_results( $prepared, ARRAY_N )[0][0] );
				}

				return $wpdb->get_results( $prepared, ARRAY_A );
				// phpcs:enable
			};

			$rows  = $query();
			$count = count( $rows );
			$total = $query( true );

			$rows = array_map(
				function( $row ) use ( $args ) {
					$result = array();

					if ( in_array( 'compatibility', $args['include'], true ) ) {
						$result[] = $row['product_id'];
						$result[] = $row['product_sku'];
					}

					$vehicle = self::instance()->get_vehicle_by_id( $row['vehicle_id'] );
					$empty   = true;

					foreach ( $this->get_vehicle_columns() as $column ) {
						$result[] = $vehicle[ 'field_' . $column ];
						$empty    = $empty && empty( $vehicle[ 'field_' . $column ] );
					}

					if ( $empty ) {
						return null;
					}

					return $result;
				},
				$rows
			);
			$rows = array_values(
				array_filter(
					$rows,
					function( $row ) {
						return null !== $row;
					}
				)
			);

			return array(
				'header' => $header,
				'rows'   => $rows,
				'limit'  => $args['limit'],
				'offset' => $args['offset'],
				'total'  => (int) $total,
				'end'    => $args['offset'] + $count,
			);
		}

		/**
		 * Imports vehicles.
		 */
		public function ajax_import() {
			if ( ! $this->get_attribute_slug() ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'You must first select the vehicle attribute.', 'redparts-sputnik' ),
					)
				);
			}

			WPML::switch_ajax_language();

			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce(
					sanitize_key( wp_unslash( $_POST['nonce'] ) ),
					'redparts_sputnik_vehicles_import'
				)
			) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'Action failed. Please refresh the page and retry.', 'redparts-sputnik' ),
					)
				);
			}

			if ( ! current_user_can( 'manage_product_terms' ) ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'You do not have permission to import vehicles.', 'redparts-sputnik' ),
					)
				);
			}

			if ( ! isset( $_POST['data']['rows'] ) ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'Rows required.', 'redparts-sputnik' ),
					)
				);
			}

			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			$errors = $this->import( wp_unslash( $_POST['data']['rows'] ) );

			wp_send_json_success(
				array(
					'errors' => $errors,
				)
			);
		}

		/**
		 * Clear vehicles.
		 */
		public function ajax_clear() {
			$attribute_slug = $this->get_attribute_slug();
			$limit          = 100;

			if ( ! $attribute_slug ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'You must first select the vehicle attribute.', 'redparts-sputnik' ),
					)
				);
			}
			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce(
					sanitize_key( wp_unslash( $_POST['nonce'] ) ),
					'redparts_sputnik_vehicles_clear'
				)
			) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'Action failed. Please refresh the page and retry.', 'redparts-sputnik' ),
					)
				);
			}

			WPML::switch_ajax_language();

			if ( ! current_user_can( 'manage_product_terms' ) ) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'You do not have permission to clear vehicles.', 'redparts-sputnik' ),
					)
				);
			}

			$exclude  = array();
			$term_all = get_term_by( 'slug', '__all__', $attribute_slug );

			if ( $term_all && ! is_wp_error( $term_all ) ) {
				$exclude[] = $term_all->term_id;
			}

			$args = array(
				'taxonomy'   => $attribute_slug,
				'hide_empty' => false,
				'exclude'    => $exclude,
			);

			$terms = get_terms(
				array_merge(
					$args,
					array( 'number' => $limit )
				)
			);

			if ( is_wp_error( $terms ) ) {
				wp_send_json_error(
					array(
						'message' => $terms->get_error_message(),
					)
				);
				return;
			}

			foreach ( $terms as $term ) {
				$result = wp_delete_term( $term->term_id, $attribute_slug );

				if ( true !== $result ) {
					wp_send_json_error(
						array(
							'message' => esc_html__( 'Failed to delete vehicle.', 'redparts-sputnik' ),
						)
					);
					return;
				}
			}

			$remaining = get_terms( array_merge( $args, array( 'fields' => 'count' ) ) );

			if ( is_wp_error( $remaining ) ) {
				wp_send_json_error(
					array(
						'message' => $remaining->get_error_message(),
					)
				);
				return;
			}

			$remaining = (int) $remaining;

			wp_send_json_success(
				array(
					'deleted'   => count( $terms ),
					'remaining' => $remaining,
				)
			);
		}

		/**
		 * Returns an expanded list of vehicle fields.
		 *
		 * @return array
		 */
		public function get_vehicle_columns(): array {
			static $fields = null;

			if ( null === $fields ) {
				$fields = array_reduce(
					self::$fields,
					function ( $result, $field ) {
						if ( 'year' === $field['type'] ) {
							$result[] = $field['slug'] . '_since';
							$result[] = $field['slug'] . '_until';
						} else {
							$result[] = $field['slug'];
						}

						return $result;
					},
					array()
				);
			}

			return $fields;
		}

		/**
		 * Sanitize rows for import.
		 *
		 * @param array $raw_rows Array of rows for import.
		 *
		 * @return array
		 */
		protected function sanitize_rows( array $raw_rows ): array {
			$raw_rows = is_array( $raw_rows ) ? array_values( $raw_rows ) : array();
			$rows     = array();
			$errors   = array();

			foreach ( $raw_rows as $index => $raw_row ) {
				$row = array(
					'index' => $index,
				);

				if ( ! empty( $raw_row['product_id'] ) ) {
					$row['product_id'] = absint( $raw_row['product_id'] );
				}
				if ( ! empty( $raw_row['product_sku'] ) ) {
					$row['product_sku'] = sanitize_text_field( $raw_row['product_sku'] );
				}

				$all_empty             = true;
				$not_enough_field      = false;
				$not_enough_field_keys = array();

				foreach ( $this->get_vehicle_columns() as $vehicle_column ) {
					$field_key = 'vehicle_' . $vehicle_column;

					if ( ! isset( $raw_row[ $field_key ] ) ) {
						$not_enough_field = true;

						if ( ! in_array( $field_key, $not_enough_field_keys, true ) ) {
							$not_enough_field_keys[] = $field_key;
						}

						break;
					}

					if ( ! empty( $raw_row[ $field_key ] ) ) {
						$all_empty = false;
					}

					$row[ $field_key ] = sanitize_text_field( $raw_row[ $field_key ] );
				}

				if ( $not_enough_field ) {
					$errors[] = array(
						'message'    => sprintf(
							// translators: %s comma-separated vehicle fields list.
							esc_html__( 'No required vehicle fields: %s.', 'redparts-sputnik' ),
							implode( ', ', $not_enough_field_keys )
						),
						'row_number' => $index,
					);
					continue;
				}
				if ( $all_empty ) {
					$errors[] = array(
						'message'    => esc_html__( 'Empty string cannot be imported.', 'redparts-sputnik' ),
						'row_number' => $index,
					);
					continue;
				}

				$rows[] = $row;
			}

			return array(
				'rows'   => $rows,
				'errors' => $errors,
			);
		}

		/**
		 * Import vehicles.
		 *
		 * @param mixed $rows Vehicle rows for import.
		 *
		 * @return array
		 */
		public function import( $rows ): array {
			$attribute_slug = $this->get_attribute_slug();

			$result = $this->sanitize_rows( $rows );
			$errors = $result['errors'];

			$db_rows = $this->get_vehicles_by_rows( $result['rows'] );

			// Group rows by product ID.
			$rows_by_product = array();

			$get_product_by_id = function( $product_id ) {
				static $cache = array();

				if ( isset( $cache[ $product_id ] ) ) {
					return $cache[ $product_id ];
				}

				$cache[ $product_id ] = wc_get_product( $product_id );

				return $cache[ $product_id ];
			};

			$get_product_by_sku = function( $product_sku ) use ( $get_product_by_id ) {
				static $cache = array();

				if ( isset( $cache[ $product_sku ] ) ) {
					return $cache[ $product_sku ];
				}

				$product    = null;
				$product_id = wc_get_product_id_by_sku( $product_sku );

				if ( $product_id ) {
					$product = $get_product_by_id( $product_id );
				}

				$cache[ $product_sku ] = $product;

				return $cache[ $product_sku ];
			};

			foreach ( $result['rows'] as $row_index => $row ) {
				$row_index = ! empty( $row['index'] ) ? $row['index'] : $row_index;
				$product   = null;

				if ( ! $product && ! empty( $row['product_id'] ) ) {
					$product = $get_product_by_id( $row['product_id'] );
				}
				if ( ! $product && ! empty( $row['product_sku'] ) ) {
					$product = $get_product_by_sku( $row['product_sku'] );
				}
				if ( ! $product && ( ! empty( $row['product_id'] ) || ! empty( $row['product_sku'] ) ) ) {
					$errors[] = array(
						'message'    => esc_html__( 'Product not found.', 'redparts-sputnik' ),
						'row_number' => $row_index,
					);
					continue;
				}

				$product_id = null === $product ? 0 : $product->get_id();

				if ( ! isset( $rows_by_product[ $product_id ] ) ) {
					$rows_by_product[ $product_id ] = array(
						'product' => $product,
						'rows'    => array(),
					);
				}

				$row['index'] = $row_index;

				$rows_by_product[ $product_id ]['rows'][] = $row;
			}

			// Import.
			foreach ( $rows_by_product as $row_container ) {
				$product = $row_container['product'];

				$product_term_ids = array();

				foreach ( $row_container['rows'] as $row ) {
					$row_index = $row['index'];

					$term_name = array();

					foreach ( self::$fields as $field ) {
						if ( 'year' === $field['type'] ) {
							$since = $row[ 'vehicle_' . $field['slug'] . '_since' ];
							$until = $row[ 'vehicle_' . $field['slug'] . '_until' ];

							$term_name[] = $since . ( $until ? '-' . $until : '' );
						} else {
							$term_name[] = $row[ 'vehicle_' . $field['slug'] ];
						}
					}

					$term_name = implode( ' ', $term_name );

					$terms = array_values(
						array_filter(
							$db_rows,
							function( $db_row ) use ( $row ) {
								foreach ( Vehicles::instance()->get_vehicle_columns() as $column ) {
									if ( $db_row[ 'redparts_' . $column ] !== $row[ 'vehicle_' . $column ] ) {
										return false;
									}
								}

								return true;
							}
						)
					);

					if ( 0 === count( $terms ) ) {
						$is_term_exists = term_exists( $term_name, $attribute_slug );

						if ( $is_term_exists ) {
							$errors[] = array(
								'message'    => esc_html__( 'A vehicle with the same name already exists.', 'redparts-sputnik' ),
								'row_number' => $row_index,
							);
							continue;
						}

						$term = wp_insert_term( $term_name, $attribute_slug );

						if ( is_wp_error( $term ) ) {
							$errors[] = array(
								'message'    => $term->get_error_message(),
								'row_number' => $row_index,
							);
							continue;
						}

						$term_id = $term['term_id'];
						$db_row  = array( 'term_id' => $term_id );

						foreach ( $this->get_vehicle_columns() as $field_column ) {
							$db_row[ 'redparts_' . $field_column ] = $row[ 'vehicle_' . $field_column ];

							update_term_meta(
								$term_id,
								'redparts_' . $field_column,
								sanitize_text_field( $row[ 'vehicle_' . $field_column ] )
							);

							$this->save_vehicle_field( $term_id, 'redparts_' . $field_column, sanitize_text_field( $row[ 'vehicle_' . $field_column ] ) );
						}

						$db_rows[] = $db_row;
					} else {
						$term_id = $terms[0]['term_id'];
					}

					$term_id = absint( $term_id );

					$product_term_ids[] = $term_id;
				}

				if ( $product && ! empty( $product_term_ids ) ) {
					$attributes = $product->get_attributes( 'edit' );
					$options    = array();

					if ( isset( $attributes[ $attribute_slug ] ) ) {
						// Change the attribute object reference to indicate that the attributes have changed.
						$attribute = clone $attributes[ $attribute_slug ];
						$options   = array_map( 'absint', $attribute->get_options() );
					} else {
						$attribute = new WC_Product_Attribute();
						$attribute->set_id( wc_attribute_taxonomy_id_by_name( $attribute_slug ) );
						$attribute->set_name( $attribute_slug );
						$attribute->set_visible( apply_filters( 'woocommerce_attribute_default_visibility', 1 ) );
						$attribute->set_variation( apply_filters( 'woocommerce_attribute_default_is_variation', 0 ) );
					}

					$attributes[ $attribute_slug ] = $attribute;

					$need_to_save = false;

					foreach ( $product_term_ids as $term_id ) {
						if ( ! in_array( $term_id, $options, true ) ) {
							$options[] = $term_id;

							$need_to_save = true;
						}
					}

					if ( $need_to_save ) {
						$attribute->set_options( $options );

						$product->set_attributes( array_merge( $attributes ) );
						$product->save();
					}
				}
			}

			$this->clear_get_options_cache();

			return $errors;
		}

		/**
		 * Returns vehicles by import rows.
		 *
		 * @param array $rows Rows to import.
		 *
		 * @return array
		 */
		protected function get_vehicles_by_rows( array $rows ): array {
			global $wpdb;

			if ( 0 === count( $rows ) ) {
				return array();
			}

			$attribute_slug = $this->get_attribute_slug();

			$sql_placeholders = array();
			$sql_select       = array( 'tt.term_id' );
			$sql_joins        = array();
			$sql_where        = array();

			foreach ( $this->get_vehicle_columns() as $column_index => $column ) {
				$table_name = 'tm' . $column_index;

				$sql_select[]       = "$table_name.meta_value AS %s";
				$sql_placeholders[] = 'redparts_' . $column;
			}

			foreach ( $this->get_vehicle_columns() as $column_index => $column ) {
				$table_name = 'tm' . $column_index;

				$sql_joins[]        = "LEFT JOIN $wpdb->termmeta AS $table_name ON tt.term_id = $table_name.term_id AND $table_name.meta_key = %s";
				$sql_placeholders[] = 'redparts_' . $column;
			}

			$sql_where[]        = 'tt.taxonomy = %s';
			$sql_placeholders[] = $attribute_slug;

			$sql_vehicles_where = array();

			foreach ( $rows as $row ) {
				$sql_vehicle_where = array();

				foreach ( $this->get_vehicle_columns() as $column_index => $column ) {
					$table_name = 'tm' . $column_index;

					$sql_vehicle_where[] = "$table_name.meta_value = %s";
					$sql_placeholders[]  = $row[ 'vehicle_' . $column ];
				}

				$sql_vehicles_where[] = '(' . implode( ' AND ', $sql_vehicle_where ) . ')';
			}

			$sql_where[] = "(\n" . implode( " OR\n", $sql_vehicles_where ) . ')';

			$sql_select = implode( ', ', $sql_select );
			$sql_joins  = implode( "\n", $sql_joins );
			$sql_where  = implode( " AND\n", $sql_where );

			$sql = "
			SELECT $sql_select
			FROM $wpdb->term_taxonomy AS tt
			$sql_joins
			WHERE
			$sql_where
			";

			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
			// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
			return $wpdb->get_results( $wpdb->prepare( $sql, ...$sql_placeholders ), ARRAY_A );
			// phpcs:enable
		}

		/**
		 * Shortcode that returns the vehicle selection control.
		 *
		 * @noinspection PhpMissingParamTypeInspection
		 *
		 * @param array $args Arguments.
		 *
		 * @return string
		 */
		public function vehicle_select_shortcode( $args = array() ): string {
			$args = shortcode_atts(
				array(
					'class'    => '',
					'location' => '',
				),
				$args
			);

			$classes = array(
				'th-vehicle-select',
			);

			if ( ! empty( $args['class'] ) ) {
				$classes[] = $args['class'];
			}
			if ( ! empty( $args['location'] ) ) {
				$classes[] = 'th-vehicle-select--location--' . $args['location'];
			}

			ob_start();

			?>
			<div
				class="<?php echo esc_attr( join( ' ', $classes ) ); ?>"
				data-ajax-url="<?php echo esc_url( apply_filters( 'redparts_sputnik_ajax_url', '' ) ); ?>"
				data-nonce="<?php echo esc_attr( wp_create_nonce( 'redparts_sputnik_vehicle_select_load_data' ) ); ?>"
			>
				<div class="th-vehicle-select__body">
					<?php foreach ( self::$fields as $index => $field ) : ?>
						<?php

						$options      = 0 === $index ? $this->get_options( array() ) : array();
						$item_classes = array( 'th-vehicle-select__item' );

						if ( 0 !== $index ) {
							$item_classes[] = 'th-vehicle-select__item--disabled';
						}

						?>
						<div
							class="<?php echo esc_attr( implode( ' ', $item_classes ) ); ?>"
							data-label="<?php echo esc_attr( $field['label'] ); ?>"
						>
							<select
								class="th-vehicle-select__item-control"
								name="<?php echo esc_attr( $field['slug'] ); ?>"
								aria-label="<?php echo esc_attr( $field['label'] ); ?>"
								<?php disabled( 0 !== $index ); ?>
							>
								<option value="null"><?php echo esc_html( $field['placeholder'] ); ?></option>
								<?php foreach ( $options as $option ) : ?>
									<option value="<?php echo esc_attr( wp_json_encode( $option['value'] ) ); ?>">
										<?php echo esc_html( $option['title'] ); ?>
									</option>
								<?php endforeach; ?>
							</select>
							<div class="th-vehicle-select__item-loader"></div>
						</div>
					<?php endforeach; ?>
				</div>
			</div>
			<?php

			return ob_get_clean();
		}

		/**
		 * Shortcode that returns a control for selecting a vehicle by VIN.
		 *
		 * @noinspection PhpMissingParamTypeInspection
		 *
		 * @param array $args Arguments.
		 *
		 * @return string
		 */
		public function vehicle_vin_shortcode( $args = array() ): string {
			$args = shortcode_atts(
				array(
					'class' => '',
				),
				$args
			);

			$classes = array(
				'th-vehicle-vin',
			);

			if ( ! empty( $args['class'] ) ) {
				$classes[] = $args['class'];
			}

			ob_start();
			?>
			<div
				class="<?php echo esc_attr( join( ' ', $classes ) ); ?>"
				data-ajax-url="<?php echo esc_url( apply_filters( 'redparts_sputnik_ajax_url', '' ) ); ?>"
				data-nonce="<?php echo esc_attr( wp_create_nonce( 'redparts_sputnik_vehicle_vin_load_data' ) ); ?>"
			>
				<div class="th-vehicle-vin__field">
					<input
						type="text"
						placeholder="<?php echo esc_attr__( 'Enter VIN number', 'redparts-sputnik' ); ?>"
						aria-label="<?php echo esc_attr__( 'VIN number', 'redparts-sputnik' ); ?>"
					>
					<div class="th-vehicle-vin__loader"></div>
				</div>
				<div class="th-vehicle-vin__alert"></div>
			</div>
			<?php

			return ob_get_clean();
		}

		/**
		 * Shortcode that returns a vehicle selection form.
		 *
		 * @noinspection PhpMissingParamTypeInspection
		 *
		 * @param array $args Arguments.
		 *
		 * @return string
		 */
		public function vehicle_form_shortcode( $args = array() ): string {
			$args = shortcode_atts(
				array(
					'class'    => '',
					'location' => '',
				),
				$args
			);

			$location = $args['location'];
			$classes  = array(
				'th-vehicle-form',
			);

			if ( ! empty( $args['class'] ) ) {
				$classes[] = $args['class'];
			}
			if ( ! empty( $location ) ) {
				$classes[] = 'th-vehicle-form--location--' . $location;
			}

			ob_start();

			?>
			<div class="<?php echo esc_attr( join( ' ', $classes ) ); ?>">
				<div class="th-vehicle-form__body">
					<?php echo do_shortcode( '[redparts_sputnik_vehicle_select class="th-vehicle-form__vehicle-select" location="' . esc_attr( $location ) . '"]' ); ?>

					<?php if ( 'yes' === Settings::instance()->get( 'search_vehicles_by_vin' ) ) : ?>
						<div class="th-vehicle-form__divider">
							<?php echo esc_html__( 'Or', 'redparts-sputnik' ); ?>
						</div>

						<?php echo do_shortcode( '[redparts_sputnik_vehicle_vin class="th-vehicle-form__vin"]' ); ?>
					<?php endif; ?>
				</div>
			</div>
			<?php

			return ob_get_clean();
		}

		/**
		 * Returns data for vehicle select controls.
		 */
		public function ajax_vehicle_select_load_data() {
			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce(
					sanitize_key( wp_unslash( $_POST['nonce'] ) ),
					'redparts_sputnik_vehicle_select_load_data'
				)
			) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'Action failed. Please refresh the page and retry.', 'redparts-sputnik' ),
					)
				);
			}

			WPML::switch_ajax_language();

			if ( ! isset( $_POST['data']['for'] ) ) {
				wp_send_json_error();
			}

			$data_for = sanitize_text_field( wp_unslash( $_POST['data']['for'] ) );

			$index = -1;

			foreach ( self::$fields as $field_index => $field ) {
				if ( $data_for === $field['slug'] ) {
					$index = $field_index;
					break;
				}
			}

			if ( -1 === $index ) {
				wp_send_json_error();
			}

			$values        = array();
			$fields_before = array_slice( self::$fields, 0, $index );

			foreach ( $fields_before as $field ) {
				if ( ! isset( $_POST['data']['values'][ $field['slug'] ] ) ) {
					wp_send_json_error();
				}

				$values[] = sanitize_text_field(
					wp_unslash( $_POST['data']['values'][ $field['slug'] ] )
				);
			}

			$options = $this->get_options( $values );

			wp_send_json_success( $options );
		}

		/**
		 * Returns vehicle by VIN.
		 */
		public function ajax_vehicle_vin_load_data() {
			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce(
					sanitize_key( wp_unslash( $_POST['nonce'] ) ),
					'redparts_sputnik_vehicle_vin_load_data'
				)
			) {
				wp_send_json_error(
					array(
						'message' => esc_html__( 'Action failed. Please refresh the page and retry.', 'redparts-sputnik' ),
					)
				);
			}

			WPML::switch_ajax_language();

			if ( ! isset( $_POST['data']['vin'] ) ) {
				wp_send_json_error();
			}

			$vin = sanitize_text_field( wp_unslash( $_POST['data']['vin'] ) );

			// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			$rows = get_terms(
				array(
					'taxonomy'   => $this->get_attribute_slug(),
					'number'     => 1,
					'hide_empty' => false,
					'meta_query' => array(
						'relation' => 'AND',
						array(
							'key'     => 'redparts_vin',
							'value'   => $vin,
							'compare' => '=',
						),
					),
				)
			);
			// phpcs:enable

			if ( $rows ) {
				$row = $rows[0];

				$vehicle = $this->get_vehicle_by_id( $row->term_id );

				wp_send_json_success(
					array(
						'vehicle' => $vehicle,
						'message' => sprintf(
							// translators: Vehicle %s name.
							esc_html__( 'Found: %s', 'redparts-sputnik' ),
							self::get_vehicle_name( $vehicle )
						),
					)
				);
			}

			wp_send_json_error(
				array(
					'message' => esc_html__( 'We were unable to find a vehicle. Perhaps you specified an incorrect VIN.', 'redparts-sputnik' ),
				)
			);
		}

		/**
		 * Clear get_options cache.
		 *
		 * @since 1.9.0
		 */
		public function clear_get_options_cache() {
			delete_transient( 'redparts_sputnik_vehicles_get_options' );
		}

		/**
		 * Returns options for the vehicle picker selects.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @param array $values An array of filled selects values.
		 *
		 * @return array
		 */
		public function get_options( array $values ): array {
			global $wpdb;

			if ( empty( $values ) ) {
				$cached_result = get_transient( 'redparts_sputnik_vehicles_get_options' );

				if ( false !== $cached_result ) {
					return $cached_result;
				}
			}

			$fields          = array_slice( self::$fields, 0, count( $values ) + 1 );
			$return_vehicles = count( $values ) === count( self::$fields ) - 1;

			$join_statements     = array();
			$join_placeholders   = array();
			$select_statements   = array();
			$select_placeholders = array();
			$where_statements    = array();
			$where_placeholders  = array();

			if ( $return_vehicles ) {
				$select_statements[] = 'tt.term_id AS id';
				$select_statements[] = 't.slug AS slug';
			}

			$where_statements[]   = "t.slug != '__all__'";
			$where_statements[]   = 'tt.taxonomy = %s';
			$where_placeholders[] = $this->get_attribute_slug();

			foreach ( $fields as $index => $field ) {
				$slug = $field['slug'];

				$last_field = count( $fields ) === $index + 1;

				if ( 'year' === $field['type'] ) {
					$tm1 = 'tm' . ( count( $join_statements ) + 1 );
					$tm2 = 'tm' . ( count( $join_statements ) + 2 );

					$join_statements[] = "LEFT JOIN {$wpdb->prefix}redparts_sputnik_vehicle_fields AS $tm1 ON tt.term_id = $tm1.term_id AND $tm1.field_key = %s";
					$join_statements[] = "LEFT JOIN {$wpdb->prefix}redparts_sputnik_vehicle_fields AS $tm2 ON tt.term_id = $tm2.term_id AND $tm2.field_key = %s";

					$join_placeholders[] = 'redparts_' . $slug . '_since';
					$join_placeholders[] = 'redparts_' . $slug . '_until';

					if ( $last_field || $return_vehicles ) {
						$select_statements[] = "$tm1.field_value AS %s";
						$select_statements[] = "IF($tm2.field_value = '', '9999', $tm2.field_value) AS %s";

						$select_placeholders[] = 'field_' . $slug . '_since';
						$select_placeholders[] = 'field_' . $slug . '_until';
					}

					$where_statements[] = "$tm1.field_value IS NOT NULL";
					$where_statements[] = "$tm1.field_value != ''";
					$where_statements[] = "$tm2.field_value IS NOT NULL";

					if ( ! $last_field ) {
						$where_statements[]   = "$tm1.field_value <= %d AND ($tm2.field_value >= %d OR $tm2.field_value = '')";
						$where_placeholders[] = $values[ $index ];
						$where_placeholders[] = $values[ $index ];
					}
				} else {
					$tm = 'tm' . ( count( $join_statements ) + 1 );

					$join_statements[]   = "LEFT JOIN {$wpdb->prefix}redparts_sputnik_vehicle_fields AS $tm ON tt.term_id = $tm.term_id AND $tm.field_key = %s";
					$join_placeholders[] = 'redparts_' . $slug;

					if ( $last_field || $return_vehicles ) {
						$select_statements[] = "$tm.field_value AS %s";

						$select_placeholders[] = 'field_' . $slug;
					}

					$where_statements[] = "$tm.field_value IS NOT NULL";
					$where_statements[] = "$tm.field_value != ''";

					if ( ! $last_field ) {
						$where_statements[]   = "$tm.field_value = %s";
						$where_placeholders[] = $values[ $index ];
					}
				}
			}

			$select_statements = implode( ", \n", $select_statements );
			$join_statements   = implode( "\n", $join_statements );
			$where_statements  = implode( " AND\n", $where_statements );

			$placeholders = array_merge(
				$select_placeholders,
				$join_placeholders,
				$where_placeholders
			);

			$sql = "
				SELECT DISTINCT
				$select_statements
				FROM $wpdb->term_taxonomy AS tt
				LEFT JOIN $wpdb->terms AS t ON tt.term_id = t.term_id
				$join_statements
				WHERE
				$where_statements
			";

			// phpcs:disable WordPress.DB.DirectDatabaseQuery
			// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
			$rows = $wpdb->get_results( $wpdb->prepare( $sql, ...$placeholders ), ARRAY_A );
			// phpcs:enable

			$last_field      = $fields[ count( $fields ) - 1 ];
			$last_field_slug = 'field_' . $last_field['slug'];

			$result = array();

			if ( ! $return_vehicles ) {
				if ( 'year' === $last_field['type'] ) {
					$years = array();

					foreach ( $rows as $row ) {
						$since = absint( $row[ $last_field_slug . '_since' ] );
						$until = absint( '9999' === $row[ $last_field_slug . '_until' ] ? gmdate( 'Y' ) : $row[ $last_field_slug . '_until' ] );

						if ( 200 < $until - $since ) {
							continue;
						}

						foreach ( range( $since, $until ) as $year ) {
							if ( ! in_array( $year, $years, true ) ) {
								$years[] = $year;
							}
						}
					}

					$result = array_map(
						function( $year ) {
							return array(
								'title' => $year,
								'value' => $year,
							);
						},
						$years
					);
				} else {
					foreach ( $rows as $row ) {
						$result[] = array(
							'title' => $row[ $last_field_slug ],
							'value' => $row[ $last_field_slug ],
						);
					}
				}
			} else {
				foreach ( $rows as $row ) {
					$vehicle      = $row;
					$field_values = array();

					foreach ( $fields as $field_idx => $field ) {
						if ( 'year' !== $field['type'] || ! isset( $values[ $field_idx ] ) ) {
							continue;
						}

						$field_values[ $field['slug'] ] = $values[ $field_idx ];
					}

					if ( 'year' === $last_field['type'] ) {
						$since = absint( $row[ $last_field_slug . '_since' ] );
						$until = absint( '9999' === $row[ $last_field_slug . '_until' ] ? gmdate( 'Y' ) : $row[ $last_field_slug . '_until' ] );

						if ( 200 < $until - $since ) {
							continue;
						}

						foreach ( range( $since, $until ) as $year ) {
							$field_values[ $last_field['slug'] ] = $year;

							$vehicle['instance_id'] = self::encode_vehicle_instance_id( $row['id'], $field_values );
							$vehicle['link']        = self::get_vehicle_link( $vehicle );

							$result[] = array(
								'title' => $year,
								'value' => $vehicle,
							);
						}
					} else {
						$vehicle['instance_id'] = self::encode_vehicle_instance_id( $row['id'], $field_values );
						$vehicle['link']        = self::get_vehicle_link( $vehicle );

						$result[] = array(
							'title' => $row[ $last_field_slug ],
							'value' => $vehicle,
						);
					}
				}

				if ( 'yes' === Settings::instance()->get( 'vehicle_picker_group_same_values', 'no' ) ) {
					$groups       = array();
					$field_values = array();

					foreach ( $fields as $field_idx => $field ) {
						if ( ! isset( $values[ $field_idx ] ) ) {
							continue;
						}

						$field_values[ $field['slug'] ] = $values[ $field_idx ];
					}

					foreach ( $result as $row ) {
						if ( ! isset( $groups[ $row['title'] ] ) ) {
							$groups[ $row['title'] ] = array(
								'title' => $row['title'],
								'value' => array(),
							);
						}

						$groups[ $row['title'] ]['value'][] = $row['value'];
					}

					$groups = array_values(
						array_map(
							function( $group ) use ( $field_values, $last_field ) {
								if ( 1 === count( $group['value'] ) ) {
									$group['value'] = $group['value'][0];
								} else {
									$field_values[ $last_field['slug'] ] = $group['title'];

									$vehicle = array(
										'id'   => array(),
										'slug' => array(),
									);

									foreach ( $group['value'] as $value ) {
										$vehicle['id'][]   = $value['id'];
										$vehicle['slug'][] = $value['slug'];

										foreach ( self::$fields as $field ) {
											$slug = $field['slug'];

											if ( 'year' === $field['type'] ) {
												$since = $value[ 'field_' . $slug . '_since' ];
												$until = (string) ( '9999' === $value[ 'field_' . $slug . '_until' ] ? gmdate( 'Y' ) : $value[ 'field_' . $slug . '_until' ] );

												$vehicle[ 'field_' . $slug . '_since' ] = min( $since, $vehicle[ 'field_' . $slug . '_since' ] ?? $since );
												$vehicle[ 'field_' . $slug . '_until' ] = max( $until, $vehicle[ 'field_' . $slug . '_until' ] ?? $until );
											} else {
												$vehicle[ 'field_' . $slug ] = $value[ 'field_' . $slug ];
											}
										}
									}

									$vehicle['instance_id'] = self::encode_vehicle_instance_id( 0, $field_values );
									$vehicle['link']        = self::get_vehicle_link( $vehicle );

									$group['value'] = $vehicle;
								}

								return $group;
							},
							$groups
						)
					);

					$result = $groups;
				}
			}

			$order = isset( $last_field['order'] ) && 'desk' === $last_field['order'] ? SORT_DESC : SORT_ASC;

			array_multisort( array_column( $result, 'title' ), $order, $result );

			if ( empty( $values ) ) {
				set_transient( 'redparts_sputnik_vehicles_get_options', $result );
			}

			return $result;
		}

		/**
		 * Returns vehicle by field values.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @since 1.10.0
		 *
		 * @param array $values Array of field values.
		 *
		 * @return array|null
		 */
		public function get_vehicle_by_field_values( array $values ): ?array {
			global $wpdb;

			$cache_key = 'redparts_sputnik_by_field_values_' . md5( wp_json_encode( $values ) );
			$vehicle   = wp_cache_get( $cache_key );

			if ( false !== $vehicle ) {
				return $vehicle;
			}

			$join_statements     = array();
			$join_placeholders   = array();
			$select_statements   = array();
			$select_placeholders = array();
			$where_statements    = array();
			$where_placeholders  = array();

			$select_statements[] = 't.slug AS slug';
			$select_statements[] = 'tt.term_id AS id';

			$where_statements[]   = "t.slug != '__all__'";
			$where_statements[]   = 'tt.taxonomy = %s';
			$where_placeholders[] = $this->get_attribute_slug();

			foreach ( self::$fields as $field ) {
				$slug = $field['slug'];

				if ( ! isset( $values[ $slug ] ) ) {
					wp_cache_set( $cache_key, null );

					return null;
				}

				if ( 'year' === $field['type'] ) {
					$tm1 = 'tm' . ( count( $join_statements ) + 1 );
					$tm2 = 'tm' . ( count( $join_statements ) + 2 );

					$join_statements[] = "LEFT JOIN {$wpdb->prefix}redparts_sputnik_vehicle_fields AS $tm1 ON tt.term_id = $tm1.term_id AND $tm1.field_key = %s";
					$join_statements[] = "LEFT JOIN {$wpdb->prefix}redparts_sputnik_vehicle_fields AS $tm2 ON tt.term_id = $tm2.term_id AND $tm2.field_key = %s";

					$join_placeholders[] = 'redparts_' . $slug . '_since';
					$join_placeholders[] = 'redparts_' . $slug . '_until';

					$select_statements[]   = "$tm1.field_value AS %s";
					$select_statements[]   = "IF($tm2.field_value = '', '9999', $tm2.field_value) AS %s";
					$select_placeholders[] = 'field_' . $slug . '_since';
					$select_placeholders[] = 'field_' . $slug . '_until';

					$where_statements[] = "$tm1.field_value IS NOT NULL";
					$where_statements[] = "$tm1.field_value != ''";
					$where_statements[] = "$tm2.field_value IS NOT NULL";

					$where_statements[]   = "$tm1.field_value <= %d AND ($tm2.field_value >= %d OR $tm2.field_value = '')";
					$where_placeholders[] = $values[ $slug ];
				} else {
					$tm = 'tm' . ( count( $join_statements ) + 1 );

					$join_statements[]   = "LEFT JOIN {$wpdb->prefix}redparts_sputnik_vehicle_fields AS $tm ON tt.term_id = $tm.term_id AND $tm.field_key = %s";
					$join_placeholders[] = 'redparts_' . $slug;

					$select_statements[]   = "$tm.field_value AS %s";
					$select_placeholders[] = 'field_' . $slug;

					$where_statements[] = "$tm.field_value IS NOT NULL";
					$where_statements[] = "$tm.field_value != ''";

					$where_statements[] = "$tm.field_value = %s";
				}

				$where_placeholders[] = $values[ $slug ];
			}
			$select_statements = implode( ", \n", $select_statements );
			$join_statements   = implode( "\n", $join_statements );
			$where_statements  = implode( " AND\n", $where_statements );

			$placeholders = array_merge(
				$select_placeholders,
				$join_placeholders,
				$where_placeholders
			);

			$sql = "
				SELECT DISTINCT
				$select_statements
				FROM $wpdb->term_taxonomy AS tt
				LEFT JOIN $wpdb->terms AS t ON tt.term_id = t.term_id
				$join_statements
				WHERE
				$where_statements
			";

			// phpcs:disable WordPress.DB.DirectDatabaseQuery
			// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
			$rows = $wpdb->get_results( $wpdb->prepare( $sql, ...$placeholders ), ARRAY_A );
			// phpcs:enable

			$vehicle = array(
				'id'   => array(),
				'slug' => array(),
			);

			foreach ( $rows as $row ) {
				$vehicle['id'][]   = absint( $row['id'] );
				$vehicle['slug'][] = $row['slug'];

				foreach ( self::$fields as $field ) {
					$slug = $field['slug'];

					if ( 'year' === $field['type'] ) {
						$since = absint( $row[ 'field_' . $slug . '_since' ] );
						$until = absint( '9999' === $row[ 'field_' . $slug . '_until' ] ? gmdate( 'Y' ) : $row[ 'field_' . $slug . '_until' ] );

						$vehicle[ 'field_' . $slug ]            = absint( $values[ $slug ] );
						$vehicle[ 'field_' . $slug . '_since' ] = min( $since, $vehicle[ 'field_' . $slug . '_since' ] ?? $since );
						$vehicle[ 'field_' . $slug . '_until' ] = max( $until, $vehicle[ 'field_' . $slug . '_until' ] ?? $until );
					} else {
						$vehicle[ 'field_' . $slug ] = $row[ 'field_' . $slug ];
					}
				}
			}

			wp_cache_set( $cache_key, $vehicle );

			return $vehicle;
		}

		/**
		 * Returns vehicle by specified field.
		 *
		 * @param string $field Name of the field by which the car will be found..
		 * @param mixed  $value Field value.
		 *
		 * @return array|null
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public function get_vehicle_by( string $field, $value ) {
			$attribute_slug = $this->get_attribute_slug();

			if ( ! $attribute_slug ) {
				return null;
			}

			$term = get_term_by( $field, $value, $attribute_slug );

			if ( ! $term ) {
				return null;
			}

			$vehicle = array(
				'id'          => $term->term_id,
				'slug'        => $term->slug,
				'vin'         => get_term_meta( $term->term_id, 'redparts_vin', true ),
				'instance_id' => self::encode_vehicle_instance_id( $term->term_id, array() ),
			);

			foreach ( self::$fields as $field ) {
				$key = 'field_' . $field['slug'];

				if ( 'year' === $field['type'] ) {
					$vehicle[ $key . '_since' ] = get_term_meta( $term->term_id, 'redparts_' . $field['slug'] . '_since', true );
					$vehicle[ $key . '_until' ] = get_term_meta( $term->term_id, 'redparts_' . $field['slug'] . '_until', true );
				} else {
					$vehicle[ $key ] = get_term_meta( $term->term_id, 'redparts_' . $field['slug'], true );
				}
			}

			return $vehicle;
		}

		/**
		 * Returns vehicle by instance ID.
		 *
		 * @since 1.7.0
		 *
		 * @param string $vehicle_instance_id Vehicle instance ID.
		 *
		 * @return array|null
		 */
		public function get_vehicle_by_instance_id( string $vehicle_instance_id ): ?array {
			$decoded_vehicle_instance_id = self::decode_vehicle_instance_id( $vehicle_instance_id );

			if ( ! $decoded_vehicle_instance_id ) {
				return null;
			}

			if ( 0 === $decoded_vehicle_instance_id['vehicle_id'] ) {
				$vehicle = $this->get_vehicle_by_field_values( $decoded_vehicle_instance_id['field_values'] );

				if ( ! $vehicle ) {
					return null;
				}

				$vehicle['instance_id'] = $vehicle_instance_id;

				return $vehicle;
			}

			$vehicle = $this->get_vehicle_by_id( $decoded_vehicle_instance_id['vehicle_id'] );

			if ( ! $vehicle ) {
				return null;
			}

			$vehicle['instance_id'] = $vehicle_instance_id;

			return $vehicle;
		}

		/**
		 * Returns vehicle by specified ID.
		 *
		 * @param int $vehicle_id Vehicle ID.
		 *
		 * @return array|null
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public function get_vehicle_by_id( int $vehicle_id ) {
			return $this->get_vehicle_by( 'id', absint( $vehicle_id ) );
		}

		/**
		 * Returns vehicle name.
		 *
		 * @param array $vehicle Vehicle.
		 *
		 * @return string
		 */
		public static function get_vehicle_name( array $vehicle ): string {
			return apply_filters(
				'redparts_sputnik_get_vehicle_name',
				self::apply_vehicle_fields( $vehicle, self::get_vehicle_name_template() )
			);
		}

		/**
		 * Returns vehicle description.
		 *
		 * @param array $vehicle Vehicle.
		 *
		 * @return string
		 */
		public static function get_vehicle_description( array $vehicle ): string {
			return apply_filters(
				'redparts_sputnik_get_vehicle_description',
				self::apply_vehicle_fields( $vehicle, self::get_vehicle_description_template() )
			);
		}

		/**
		 * Returns the vehicle name template.
		 *
		 * @return string
		 */
		public static function get_vehicle_name_template(): string {
			return apply_filters(
				'redparts_sputnik_get_vehicle_name_template',
				Settings::instance()->get( 'vehicle_name', '' )
			);
		}

		/**
		 * Returns the vehicle name template.
		 *
		 * @return string
		 */
		public static function get_vehicle_description_template(): string {
			return apply_filters(
				'redparts_sputnik_get_vehicle_description_template',
				Settings::instance()->get( 'vehicle_description', '' )
			);
		}

		/**
		 * Applies vehicle field values to the template.
		 *
		 * @param array  $vehicle  - Vehicle data.
		 * @param string $template - Template.
		 *
		 * @return string
		 */
		private static function apply_vehicle_fields( array $vehicle, string $template ): string {
			$field_values = array();

			if ( isset( $vehicle['instance_id'] ) ) {
				$vehicle_instance_id = self::decode_vehicle_instance_id( $vehicle['instance_id'] );

				if ( $vehicle_instance_id ) {
					$field_values = $vehicle_instance_id['field_values'];
				}
			}

			foreach ( self::$fields as $field ) {
				if ( 'year' === $field['type'] ) {
					$year = $vehicle[ 'field_' . $field['slug'] . '_since' ];

					if ( isset( $field_values[ $field['slug'] ] ) ) {
						$year = $field_values[ $field['slug'] ];
					}

					$since = $vehicle[ 'field_' . $field['slug'] . '_since' ];
					$until = $vehicle[ 'field_' . $field['slug'] . '_until' ];
					$until = empty( $until ) ? gmdate( 'Y' ) : $until;
					$range = ! empty( $since ) ? $since . '–' . $until : '';

					$template = str_replace( '%' . $field['slug'] . '%', $year, $template );
					$template = str_replace( '%' . $field['slug'] . '_since%', $since, $template );
					$template = str_replace( '%' . $field['slug'] . '_until%', $until, $template );
					$template = str_replace( '%' . $field['slug'] . '_range%', $range, $template );
				} else {
					$template = str_replace( '%' . $field['slug'] . '%', $vehicle[ 'field_' . $field['slug'] ], $template );
				}
			}

			return $template;
		}

		/**
		 * Returns True if the vehicle name contains *_range (or simultaneously *_since and *_until) placeholder for
		 * the field with the "Year" type.
		 *
		 * @since 1.13.0
		 *
		 * @return bool
		 */
		public static function is_vehicle_name_contains_years_range(): bool {
			$template = self::get_vehicle_name_template();
			$result   = false;

			foreach ( self::$fields as $field ) {
				if ( 'year' !== $field['type'] ) {
					continue;
				}

				$contains_since = false !== strpos( $template, '%' . $field['slug'] . '_since%' );
				$contains_until = false !== strpos( $template, '%' . $field['slug'] . '_until%' );
				$contains_range = false !== strpos( $template, '%' . $field['slug'] . '_range%' );

				$result = $result || $contains_range || ( $contains_since && $contains_until );
			}
			return $result;
		}

		/**
		 * Encodes vehicle instance ID.
		 *
		 * @since 1.7.0
		 *
		 * @param int   $vehicle_id   Vehicle ID.
		 * @param array $field_values Vehicle field values.
		 *
		 * @return string
		 */
		private static function encode_vehicle_instance_id( int $vehicle_id, array $field_values ): string {
			$vehicle_instance_id = ( (string) $vehicle_id ) . ':';

			if ( ! empty( $field_values ) ) {
				$parts = array();

				foreach ( $field_values as $key => $value ) {
					$parts[] = $key . ':' . $value;
				}

				// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
				$vehicle_instance_id .= base64_encode( implode( ',', $parts ) );
			}

			return $vehicle_instance_id;
		}

		/**
		 * Decodes vehicle instance ID.
		 *
		 * @since 1.7.0
		 *
		 * @param string $vehicle_instance_id Vehicle instance ID.
		 *
		 * @return array|null
		 */
		private static function decode_vehicle_instance_id( string $vehicle_instance_id ): ?array {
			$vehicle_instance_id = explode( ':', $vehicle_instance_id );

			if ( 0 === count( $vehicle_instance_id ) ) {
				return null;
			}

			$vehicle_id   = absint( $vehicle_instance_id[0] );
			$field_values = array();

			if ( 2 === count( $vehicle_instance_id ) ) {
				// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
				$parts = explode( ',', base64_decode( $vehicle_instance_id[1] ) );

				foreach ( $parts as $part ) {
					$key_value = explode( ':', $part );

					if ( 2 !== count( $key_value ) ) {
						continue;
					}

					$field_values[ $key_value[0] ] = $key_value[1];
				}
			}

			return array(
				'vehicle_id'   => $vehicle_id,
				'field_values' => $field_values,
			);
		}

		/**
		 * Returns the currently filtered vehicle.
		 *
		 * @since 1.4.0
		 *
		 * @return array|null
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public static function get_filtered_vehicle() {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return null;
			}

			$attribute_filter_name = self::instance()->get_attribute_filter_name();

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			if ( isset( $_GET[ 'redparts_' . $attribute_filter_name ] ) ) {
				$vehicle_personal_id = sanitize_text_field( wp_unslash( $_GET[ 'redparts_' . $attribute_filter_name ] ) );

				return self::instance()->get_vehicle_by_instance_id( $vehicle_personal_id );
			}
			// phpcs:enable

			$attribute_slug     = self::instance()->get_attribute_slug();
			$product_attributes = WC_Query::get_layered_nav_chosen_attributes();

			if ( isset( $product_attributes[ $attribute_slug ] ) && 1 === count( $product_attributes[ $attribute_slug ]['terms'] ) ) {
				$vehicle_slug = $product_attributes[ $attribute_slug ]['terms'][0];

				return self::instance()->get_vehicle_by( 'slug', $vehicle_slug );
			}

			return null;
		}

		/**
		 * Returns the name of the currently filtered vehicle.
		 *
		 * @since 1.4.0
		 *
		 * @return string
		 */
		public static function get_filtered_vehicle_name(): string {
			$vehicle = self::get_filtered_vehicle();

			if ( ! $vehicle ) {
				return '';
			}

			return self::get_vehicle_name( $vehicle );
		}

		/**
		 * Returns vehicle link.
		 *
		 * @since 1.4.0
		 *
		 * @param string|array $vehicle Vehicle or vehicle slug.
		 *
		 * @return string
		 */
		public static function get_vehicle_link( $vehicle ): string {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return '';
			}

			$query_arg_name  = self::instance()->get_attribute_filter_name();
			$query_arg_value = self::instance()->get_attribute_filter_value( $vehicle );

			return add_query_arg(
				array(
					$query_arg_name => $query_arg_value,
				),
				get_permalink( wc_get_page_id( 'shop' ) )
			);
		}

		/**
		 * Returns true if the current page is a vehicle page.
		 *
		 * @since 1.4.0
		 *
		 * @return bool
		 */
		public static function is_vehicle(): bool {
			$filtered_vehicle_name = self::get_filtered_vehicle_name();

			return (
				class_exists( 'WooCommerce' ) &&
				! empty( $filtered_vehicle_name ) &&
				is_shop() &&
				(
					! is_search() ||
					( is_search() && empty( get_search_query() ) )
				)
			);
		}

		/**
		 * Removes all vehicle fields after deleting a term.
		 *
		 * @since 1.10.0
		 *
		 * @param int $vehicle_id Vehicle ID.
		 */
		public function on_delete_term( int $vehicle_id ) {
			global $wpdb;

			// phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->delete( $wpdb->prefix . 'redparts_sputnik_vehicle_fields', array( 'term_id' => $vehicle_id ) );
			// phpcs:enable
		}

		/**
		 * Adds or updates a vehicle field.
		 *
		 * @since 1.10.0
		 *
		 * @param int    $vehicle_id Vehicle ID.
		 * @param string $key        Field name.
		 * @param string $value      Field value.
		 *
		 * @return bool
		 */
		public function save_vehicle_field( int $vehicle_id, string $key, string $value ): bool {
			global $wpdb;

			// phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
			$result = $wpdb->replace(
				$wpdb->prefix . 'redparts_sputnik_vehicle_fields',
				array(
					'term_id'     => $vehicle_id,
					'field_key'   => $key,
					'field_value' => $value,
				)
			);
			// phpcs:enable

			if ( ! $result ) {
				return false;
			}

			return true;
		}

		/**
		 * Clear and refill vehicle fields table.
		 *
		 * @since 1.10.0
		 */
		public function reindex_vehicle_fields_table() {
			global $wpdb;

			// phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
			// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->query( "TRUNCATE TABLE {$wpdb->prefix}redparts_sputnik_vehicle_fields" );

			$values = array();

			foreach ( self::$fields as $field ) {
				if ( 'year' === $field['type'] ) {
					$values[] = 'redparts_' . $field['slug'] . '_since';
					$values[] = 'redparts_' . $field['slug'] . '_until';

				} else {
					$values[] = 'redparts_' . $field['slug'];
				}
			}

			$placeholders = implode(
				', ',
				array_map(
					function() {
						return '%s';
					},
					$values
				)
			);

			$sql = "
				INSERT IGNORE INTO {$wpdb->prefix}redparts_sputnik_vehicle_fields
				SELECT
			       tm.term_id,
			       tm.meta_key,
			       tm.meta_value
				FROM $wpdb->termmeta AS tm
				WHERE
					tm.meta_key IN ($placeholders) AND
			        tm.meta_value IS NOT NULL
			";

			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			$wpdb->query( $wpdb->prepare( $sql, ...$values ) );
			// phpcs:enable
		}
	}
}
