<?php
/**
 * This file contains code related to the brands only.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Megamenu;

use RedParts\Sputnik\Singleton;
use RedParts\Sputnik\WPML\WPML;
use WP_Post;
use stdClass;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Megamenu\Megamenu' ) ) {
	/**
	 * Class Megamenu
	 */
	class Megamenu extends Singleton {
		/**
		 * Initialization.
		 */
		public function init() {
			add_filter( 'wp_edit_nav_menu_walker', array( $this, 'walker' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
			add_action( 'admin_head-nav-menus.php', array( $this, 'register_metabox' ) );
			add_filter( 'wp_setup_nav_menu_item', array( $this, 'setup_menu_item' ) );
			add_action( 'wp_update_nav_menu_item', array( $this, 'update_menu_item' ), 10, 2 );
			add_filter( 'nav_menu_css_class', array( $this, 'menu_item_classes' ), 10, 4 );
			add_filter( 'walker_nav_menu_start_el', array( $this, 'menu_item' ), 10, 4 );
			add_filter( 'nav_menu_item_args', array( $this, 'menu_item_args' ), 10, 3 );

			if ( wp_doing_ajax() ) {
				add_action( 'wp_ajax_redparts_sputnik_megamenu', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_megamenu', array( $this, 'ajax' ) );
			}
		}

		/**
		 * Returns walker class name.
		 *
		 * @return string
		 */
		public function walker(): string {
			return '\\RedParts\\Sputnik\\Megamenu\\Walker';
		}

		/**
		 * Enqueue admin scripts.
		 *
		 * @param string $hook Current page.
		 */
		public function enqueue_admin_scripts( string $hook ) {
			if ( 'nav-menus.php' !== $hook ) {
				return;
			}

			wp_enqueue_media();
			wp_enqueue_script( 'redparts-sputnik-megamenu' );
		}

		/**
		 * Outputs megamenus via AJAX.
		 *
		 * @since 1.8.0
		 */
		public function ajax() {
			WPML::switch_ajax_language();

			// phpcs:disable WordPress.Security.NonceVerification.Missing
			if ( ! isset( $_POST['data']['ids'] ) ) {
				wp_send_json_error();
			}

			$menu_item_ids = wp_parse_id_list( wp_unslash( $_POST['data']['ids'] ) );
			$megamenus     = array();
			// phpcs:enable

			foreach ( $menu_item_ids as $menu_item_id ) {
				$menu_item = get_post( $menu_item_id );

				if ( ! $menu_item ) {
					continue;
				}

				$megamenu_menu_id = get_post_meta( $menu_item_id, '_redparts_megamenu_menu_id', true );
				$megamenu_size    = get_post_meta( $menu_item_id, '_redparts_megamenu_size', true );
				$megamenu_image   = get_post_meta( $menu_item_id, '_redparts_megamenu_image', true );

				if ( ! $megamenu_menu_id ) {
					continue;
				}

				$megamenus[ $menu_item_id ] = $this->get_megamenu(
					absint( $megamenu_menu_id ),
					$menu_item_id,
					$megamenu_size,
					$megamenu_image
				);
			}

			wp_send_json_success(
				array(
					'megamenus' => $megamenus,
				)
			);
		}

		/**
		 * Register nav menu metabox.
		 *
		 * @noinspection PhpUnused
		 */
		public function register_metabox() {
			add_meta_box(
				'redparts_sputnik_megamenu',
				esc_html__( 'RedParts Megamenu', 'redparts-sputnik' ),
				array( $this, 'metabox' ),
				'nav-menus',
				'side',
				'high'
			);
		}

		/**
		 * Outputs nav menu metabox.
		 */
		public function metabox() {
			global $nav_menu_selected_id;

			?>
			<div id="tabs-panel-list-files" class="tabs-panel tabs-panel-active">
				<p class="button-controls">
					<button
						id="redparts-megamenu-add-column"
						class="button-secondary"
						type="button"
						<?php wp_nav_menu_disabled_check( $nav_menu_selected_id ); ?>
					>
						<?php esc_html_e( 'Add Column', 'redparts-sputnik' ); ?>
					</button>
					<span class="spinner"></span>
				</p>
			</div>
			<?php
		}

		/**
		 * Change label of menu item with type redparts_megamenu_column.
		 *
		 * @param WP_Post|stdClass $menu_item - Menu item object.
		 * @return WP_Post|stdClass
		 */
		public function setup_menu_item( $menu_item ) {
			if ( 'redparts_megamenu_column' !== $menu_item->type ) {
				return $menu_item;
			}

			$menu_item->type_label = esc_html__( 'Column', 'redparts-sputnik' );

			return $menu_item;
		}

		/**
		 * Saves megamenu item properties.
		 *
		 * @param int $menu_id         ID of the updated menu.
		 * @param int $menu_item_db_id ID of the updated menu item.
		 * @noinspection  PhpUnusedParameterInspection
		 */
		public function update_menu_item( int $menu_id, int $menu_item_db_id ) {
			if ( ! current_user_can( 'edit_theme_options' ) ) {
				return;
			}

			// Add new menu item via ajax.
			if (
				isset( $_REQUEST['menu-settings-column-nonce'] ) &&
				wp_verify_nonce( sanitize_key( wp_unslash( $_REQUEST['menu-settings-column-nonce'] ) ), 'add-menu_item' )
			) {
				if (
					! isset( $_POST['menu-item']['-1']['menu-item-redparts-type'] ) ||
					'redparts_megamenu_column' !== $_POST['menu-item']['-1']['menu-item-redparts-type']
				) {
					return;
				}

				update_post_meta( $menu_item_db_id, '_menu_item_type', 'redparts_megamenu_column' );
			}

			$valid_sizes = array(
				'sm',
				'nl',
				'md',
				'lg',
				'xl',
			);

			$valid_column_sizes = array(
				'1',
				'2',
				'3',
				'4',
				'5',
				'6',
				'7',
				'8',
				'9',
				'10',
				'11',
				'12',
				'1of5',
				'2of5',
				'3of5',
				'4of5',
				'5of5',
			);

			// Update settings for existing menu items.
			if (
				isset( $_REQUEST['update-nav-menu-nonce'] ) &&
				wp_verify_nonce( sanitize_key( wp_unslash( $_REQUEST['update-nav-menu-nonce'] ) ), 'update-nav_menu' )
			) {

				if ( isset( $_POST['menu-item-redparts-megamenu-column-size'][ $menu_item_db_id ] ) ) {
					$megamenu_column_size = sanitize_text_field( wp_unslash( $_POST['menu-item-redparts-megamenu-column-size'][ $menu_item_db_id ] ) );
					$megamenu_column_size = in_array( $megamenu_column_size, $valid_column_sizes, true ) ? $megamenu_column_size : '1';

					update_post_meta( $menu_item_db_id, '_redparts_megamenu_column_size', $megamenu_column_size );
				}

				if ( isset( $_POST['menu-item-redparts-megamenu-menu-id'][ $menu_item_db_id ] ) ) {
					$megamenu_menu_id = absint( $_POST['menu-item-redparts-megamenu-menu-id'][ $menu_item_db_id ] );

					if ( $megamenu_menu_id && wp_get_nav_menu_object( $megamenu_menu_id ) ) {
						update_post_meta( $menu_item_db_id, '_redparts_megamenu_menu_id', $megamenu_menu_id );
					} else {
						delete_post_meta( $menu_item_db_id, '_redparts_megamenu_menu_id' );
					}
				}

				if ( isset( $_POST['menu-item-redparts-megamenu-size'][ $menu_item_db_id ] ) ) {
					$megamenu_size = sanitize_text_field( wp_unslash( $_POST['menu-item-redparts-megamenu-size'][ $menu_item_db_id ] ) );
					$megamenu_size = in_array( $megamenu_size, $valid_sizes, true ) ? $megamenu_size : 'nl';

					update_post_meta( $menu_item_db_id, '_redparts_megamenu_size', sanitize_key( $megamenu_size ) );
				}

				if ( isset( $_POST['menu-item-redparts-megamenu-image'][ $menu_item_db_id ] ) ) {
					$megamenu_image = absint( $_POST['menu-item-redparts-megamenu-image'][ $menu_item_db_id ] );

					if ( $megamenu_image && 'attachment' === get_post_type( $megamenu_image ) ) {
						update_post_meta( $menu_item_db_id, '_redparts_megamenu_image', $megamenu_image );
					} else {
						delete_post_meta( $menu_item_db_id, '_redparts_megamenu_image' );
					}
				}
			}
		}

		/**
		 * Adds redparts_has_megamenu to the menu item args.
		 *
		 * @noinspection PhpUnused,PhpMissingParamTypeInspection
		 *
		 * @param stdClass $args  An object of wp_nav_menu() arguments.
		 * @param WP_Post  $item  Menu item data object.
		 * @param int      $depth Depth of menu item. Used for padding.
		 *
		 * @return stdClass
		 */
		public function menu_item_args( stdClass $args, $item, int $depth ): stdClass {
			if ( 0 === $depth && get_post_meta( $item->ID, '_redparts_megamenu_menu_id', true ) ) {
				$args = (object) (array) $args;

				$args->redparts_has_megamenu = true;
			}

			return $args;
		}

		/**
		 * Filters a menu item's starting output.
		 *
		 * @param string           $item_output The menu item's starting HTML output.
		 * @param WP_Post|stdClass $item        Menu item data object.
		 * @param int              $depth       Depth of menu item. Used for padding.
		 * @param stdClass         $args        An object of wp_nav_menu() arguments.
		 * @return string
		 */
		public function menu_item( string $item_output, $item, int $depth, stdClass $args ): string {
			static $megamenu_depth = 0;

			if ( 'redparts_megamenu_column' === $item->type ) {
				return '';
			}

			if (
				0 === $megamenu_depth &&
				0 === $depth &&
				! empty( $args->redparts_megamenu ) &&
				true === $args->redparts_megamenu
			) {
				$megamenu_depth++;

				$megamenu_menu_id = get_post_meta( $item->ID, '_redparts_megamenu_menu_id', true );
				$megamenu_size    = get_post_meta( $item->ID, '_redparts_megamenu_size', true );
				$megamenu_image   = get_post_meta( $item->ID, '_redparts_megamenu_image', true );
				$megamenu_args    = ! empty( $args->redparts_megamenu_args ) ? $args->redparts_megamenu_args : array();
				$megamenu_lazy    = (bool) ( $args->redparts_lazy_megamenu ?? false );

				if ( $megamenu_menu_id ) {
					$item_output .= $this->get_megamenu(
						absint( $megamenu_menu_id ),
						$item->ID,
						$megamenu_size,
						$megamenu_image,
						$megamenu_args,
						$megamenu_lazy
					);
				}

				$megamenu_depth--;
			}

			return $item_output;
		}

		/**
		 * Returns an array of CSS classes of the menu item.
		 *
		 * @param array            $classes Array of css classes.
		 * @param WP_Post|stdClass $item    Menu item data object.
		 * @param stdClass         $args    An object of wp_nav_menu() arguments.
		 * @param int              $depth   Depth of menu item. Used for padding.
		 * @return array
		 * @noinspection PhpUnusedParameterInspection
		 */
		public function menu_item_classes( array $classes, $item, stdClass $args, int $depth ): array {
			if ( 0 === $depth ) {
				$megamenu_menu_id = get_post_meta( $item->ID, '_redparts_megamenu_menu_id', true );

				if ( $megamenu_menu_id && wp_get_nav_menu_object( $megamenu_menu_id ) ) {
					$classes[] = 'menu-item-has-children';
					$classes[] = 'menu-item-megamenu';
				}
			}

			if ( 'redparts_megamenu_column' === $item->type ) {
				$megamenu_column_size = get_post_meta( $item->ID, '_redparts_megamenu_column_size', true );

				$classes[] = sanitize_html_class( 'th-col-' . $megamenu_column_size );
			}

			return array_filter( $classes );
		}

		/**
		 * Returns rendered megamenu.
		 *
		 * @since 1.8.0 Added $lazy argument.
		 *
		 * @param int    $menu_id      Megamenu ID.
		 * @param int    $menu_item_id Menu item ID.
		 * @param string $size         Size of megamenu.
		 * @param string $image        Background image.
		 * @param array  $args         Megamenu menu args.
		 * @param bool   $lazy         Indicates that menu should be lazy.
		 *
		 * @return string
		 * @noinspection PhpMissingParamTypeInspection
		 */
		public function get_megamenu( int $menu_id, int $menu_item_id, string $size, string $image, $args = array(), $lazy = false ): string {
			if ( $lazy ) {
				return '<div class="th-megamenu th-megamenu--lazy ' . sanitize_html_class( 'th-megamenu--size--' . $size ) . '" data-id="' . esc_attr( $menu_item_id ) . '"></div>';
			}

			ob_start();

			$menu = wp_get_nav_menu_object( $menu_id );

			if ( $menu ) {
				$image      = wp_get_attachment_image_url( $image, 'full' );
				$image_html = '';

				if ( $image ) {
					$image_html = '<div class="th-megamenu__image"><img src="' . esc_url( $image ) . '" alt=""></div>';
				}

				wp_nav_menu(
					array_merge(
						array(
							'menu'            => $menu_id,
							'menu_class'      => 'th-megamenu__row th-row',
							'container_class' => 'th-megamenu ' . sanitize_html_class( 'th-megamenu--size--' . $size ),
							'items_wrap'      => $image_html . '<ul id="%1$s" class="%2$s">%3$s</ul>',
							'depth'           => 4,
						),
						$args
					)
				);
			}

			return ob_get_clean();
		}
	}
}
