<?php
/**
 * Block features widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Features' ) ) {
	/**
	 * Class Block_Features
	 */
	class Block_Features extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_features',
				'description'                 => esc_html_x( 'Displays shop features.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_features',
				esc_html_x( 'RedParts: Block Features', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Returns pre-defined icons.
		 *
		 * @return array
		 */
		protected function icons(): array {
			$icons = array(
				'fi-free-delivery-48'    => array(
					'label' => esc_html__( 'Free Shipping', 'redparts-sputnik' ),
					'code'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 48 48" width="48" height="48">
<path d="M44.6,26.9l-1.2-5c0.3-0.1,0.6-0.4,0.6-0.7v-0.8c0-1.7-1.4-3.2-3.2-3.2h-5.7v-1.7c0-0.9-0.7-1.6-1.6-1.6H23.1l6.4-2.6
	c0.4-0.2,0.6-0.6,0.4-1c-0.2-0.4-0.6-0.6-1-0.4l-5.2,2.1c1.6-1,3.2-2.2,3.8-2.9c1.2-1.5,0.9-3.7-0.7-4.9c-1.5-1.2-3.7-0.9-4.9,0.7
	l0,0c-0.9,1.1-2,4.3-2.7,6.5c-0.7-2.2-1.9-5.4-2.7-6.5l0,0c-1.2-1.5-3.4-1.8-4.9-0.7C10,5.5,9.7,7.7,10.9,9.2
	c0.6,0.8,2.2,1.9,3.8,2.9l-5.2-2.1c-0.4-0.2-0.8,0-1,0.4c-0.2,0.4,0,0.8,0.4,1l6.4,2.6H4.8c-0.9,0-1.6,0.7-1.6,1.6v13.6
	C3.2,29.6,3.5,30,4,30c0.4,0,0.8-0.3,0.8-0.8V15.6c0,0,0,0,0,0h28.9c0,0,0,0,0,0v13.6c0,0.4,0.3,0.8,0.8,0.8c0.4,0,0.8-0.3,0.8-0.8
	v-0.9H44c0,0,0,0,0,0c0,0,0,0,0,0c1.1,0,2,0.7,2.3,1.7H44c-0.4,0-0.8,0.3-0.8,0.8v1.6c0,1.3,1.1,2.4,2.4,2.4h0.9v3.3h-2
	c-0.6-1.9-2.4-3.2-4.5-3.2c-2.1,0-3.9,1.3-4.5,3.2h-0.4v-5.7c0-0.4-0.3-0.8-0.8-0.8c-0.4,0-0.8,0.3-0.8,0.8v5.7H18.1
	c-0.6-1.9-2.4-3.2-4.5-3.2c-2.1,0-3.9,1.3-4.5,3.2H4.8c0,0,0,0,0,0v-1.7H8c0.4,0,0.8-0.3,0.8-0.8S8.4,34.9,8,34.9H0.8
	c-0.4,0-0.8,0.3-0.8,0.8s0.3,0.8,0.8,0.8h2.5V38c0,0.9,0.7,1.6,1.6,1.6h4.1c0,0,0,0,0,0c0,2.6,2.1,4.8,4.8,4.8s4.8-2.1,4.8-4.8
	c0,0,0,0,0,0h16.9c0,0,0,0,0,0c0,2.6,2.1,4.8,4.8,4.8s4.8-2.1,4.8-4.8c0,0,0,0,0,0h2.5c0.4,0,0.8-0.3,0.8-0.8v-8
	C48,28.8,46.5,27.2,44.6,26.9z M23.1,5.9L23.1,5.9c0.7-0.9,1.9-1,2.8-0.4s1,1.9,0.4,2.8c-0.3,0.3-1.1,1.2-4.1,3
	c-0.6,0.4-1.2,0.7-1.7,1C21.2,10.1,22.4,6.9,23.1,5.9z M12.1,8.3c-0.7-0.9-0.5-2.1,0.4-2.8c0.4-0.3,0.8-0.4,1.2-0.4
	c0.6,0,1.2,0.3,1.6,0.8l0,0c0.7,1,1.9,4.2,2.6,6.5c-0.5-0.3-1.1-0.6-1.7-1C13.2,9.5,12.4,8.7,12.1,8.3z M35.2,21.9h6.7l1.2,4.9h-7.9
	V21.9z M40.8,18.7c0.9,0,1.7,0.7,1.7,1.7v0h-7.3v-1.7L40.8,18.7L40.8,18.7z M13.6,42.9c-1.8,0-3.3-1.5-3.3-3.3s1.5-3.3,3.3-3.3
	s3.3,1.5,3.3,3.3S15.4,42.9,13.6,42.9z M40,42.9c-1.8,0-3.3-1.5-3.3-3.3s1.5-3.3,3.3-3.3s3.3,1.5,3.3,3.3S41.8,42.9,40,42.9z
	 M45.6,33.3c-0.5,0-0.9-0.4-0.9-0.9v-0.9h1.7v1.7L45.6,33.3L45.6,33.3z"></path>
<path d="M13.6,38.1c-0.9,0-1.6,0.7-1.6,1.6s0.7,1.6,1.6,1.6s1.6-0.7,1.6-1.6S14.4,38.1,13.6,38.1z"></path>
<path d="M40,38.1c-0.9,0-1.6,0.7-1.6,1.6s0.7,1.6,1.6,1.6c0.9,0,1.6-0.7,1.6-1.6S40.9,38.1,40,38.1z"></path>
<path d="M19.2,35.6c0,0.4,0.3,0.8,0.8,0.8h11.2c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8H20C19.6,34.9,19.2,35.2,19.2,35.6z"></path>
<path d="M2.4,33.2H12c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8H2.4c-0.4,0-0.8,0.3-0.8,0.8S1.9,33.2,2.4,33.2z"></path>
<path d="M12,21.9c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8H8.8c-0.4,0-0.8,0.3-0.8,0.8v6.4c0,0.4,0.3,0.8,0.8,0.8
	c0.4,0,0.8-0.3,0.8-0.8v-2.5h1.7c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8H9.5v-1.7L12,21.9L12,21.9z"></path>
<path d="M19.1,23.2c0-1.5-1.2-2.8-2.8-2.8h-2c-0.4,0-0.8,0.3-0.8,0.8v6.4c0,0.4,0.3,0.8,0.8,0.8c0.4,0,0.8-0.3,0.8-0.8V26h1.3
	l1.4,2.1c0.1,0.2,0.4,0.3,0.6,0.3c0.1,0,0.3,0,0.4-0.1c0.3-0.2,0.4-0.7,0.2-1l-1.1-1.7C18.6,25,19.1,24.2,19.1,23.2z M15.1,21.9h1.3
	c0.7,0,1.3,0.6,1.3,1.3s-0.6,1.3-1.3,1.3h-1.3V21.9z"></path>
<path d="M24,21.9c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8h-3.2c-0.4,0-0.8,0.3-0.8,0.8v6.4c0,0.4,0.3,0.8,0.8,0.8H24
	c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8h-2.5v-1.7c0,0,0,0,0,0h1.6c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8h-1.6c0,0,0,0,0,0v-1.7
	L24,21.9L24,21.9z"></path>
<path d="M29.6,21.9c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8h-3.2c-0.4,0-0.8,0.3-0.8,0.8v6.4c0,0.4,0.3,0.8,0.8,0.8h3.2
	c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8h-2.5v-1.7H28c0.4,0,0.8-0.3,0.8-0.8s-0.3-0.8-0.8-0.8h-0.9v-1.7L29.6,21.9L29.6,21.9z"></path>
</svg>',
				),
				'fi-24-hours-48'         => array(
					'label' => esc_html__( '24 Hours', 'redparts-sputnik' ),
					'code'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 48 48" width="48" height="48">
<g>
	<path d="M46.2,18.2H46v-0.9c0-1.2-1.2-1.8-2.5-1.8c-0.5,0-1,0.1-1.4,0.3c0,0,2.3-4.6,2.3-4.6c1-2.3-3-4-4.4-1.4l-5.2,10.2
		c-0.3,0.6-0.4,1.4,0,2.1c-0.2-0.1-0.4-0.2-0.7-0.2h-4.9c1.6-1.6,6.6-3.5,6.6-8c0-1.6-0.7-3-1.9-4c-1.1-0.9-2.5-1.4-4-1.4
		c-2.8,0-5.8,1.8-5.8,5c0,2.2,1.4,2.5,2.3,2.5c1.7,0,2.5-1,2.5-2c0-0.5,0.1-1,1-1c0.9,0,1,0.6,1,0.9c0,2.3-7.1,4.5-7.1,8.4v2.2
		c0,1.1,1.3,1.8,2.2,1.8h8c1.4,0,2.3-2.2,1.3-3.7c0.3,0.1,0.6,0.2,0.9,0.2H41v1.7c0,1,1,1.8,2.5,1.8s2.5-0.7,2.5-1.8v-1.7h0.3
		c1,0,1.8-1.2,1.8-2.3C48,19.1,47.3,18.2,46.2,18.2L46.2,18.2z M34.2,24.9h-8c-0.4,0-0.8-0.3-0.8-0.4v-2.2c0-3.1,7.1-5,7.1-8.4
		c0-1.1-0.8-2.3-2.4-2.3c-1.5,0-2.4,0.9-2.4,2.4c0,0.2-0.4,0.6-1.1,0.6c-0.5,0-0.9-0.1-0.9-1.1c0-2.3,2.3-3.6,4.4-3.6
		c2.3,0,4.5,1.4,4.5,3.9c0,4.5-6.9,5.8-6.9,8.5v0.2c0,0.4,0.3,0.7,0.7,0.7h5.9c0.1,0,0.4,0.3,0.4,0.8C34.6,24.5,34.3,24.8,34.2,24.9
		L34.2,24.9z M46.2,21.4h-0.9c-0.4,0-0.7,0.3-0.7,0.7v2.4c0,0.2-0.4,0.4-1,0.4s-1-0.2-1-0.4v-2.4c0-0.4-0.3-0.7-0.7-0.7h-5.3
		c-0.5,0-0.5-0.5-0.4-0.8c0,0,0,0,0,0l5.2-10.2c0.5-1,1.9-0.2,1.9,0.1c0,0.1,0,0-4,8.1c-0.2,0.5,0.1,1,0.6,1h1.9
		c0.4,0,0.7-0.3,0.7-0.7v-1.6c0-0.1,0.4-0.4,1.1-0.4c0.6,0,1,0.2,1,0.4v1.6c0,0.4,0.3,0.7,0.7,0.7h1c0.2,0,0.4,0.4,0.4,0.9
		C46.6,21,46.3,21.3,46.2,21.4L46.2,21.4z M41,18v0.2H41L41,18z"></path>
	<g>
		<path d="M37,36.6c2.4-2.3,4.2-5.2,5.2-8.4c0.1-0.4-0.1-0.8-0.5-0.9c-0.4-0.1-0.8,0.1-0.9,0.5c-0.9,2.9-2.6,5.6-4.8,7.7
			c-0.3,0.3-0.3,0.7,0,1C36.2,36.9,36.7,36.9,37,36.6L37,36.6z"></path>
		<path d="M35.1,7.9c0.2-0.3,0.2-0.8-0.2-1c-5.3-3.7-11.8-4.5-17.6-2.5C17,4.5,16.8,4.9,17,5.3s0.5,0.6,0.9,0.4
			C23.1,3.9,29.2,4.5,34.1,8C34.4,8.2,34.9,8.2,35.1,7.9L35.1,7.9z"></path>
		<path d="M25.2,43.6c-0.9-0.5-1.9-0.6-2.8-0.4c-0.5,0.1-1.1,0.2-1.6,0.3c-0.4,0-0.7,0.4-0.6,0.8s0.4,0.7,0.8,0.6
			c0.6,0,1.2-0.1,1.8-0.3c0.6-0.1,1.2,0,1.8,0.3c1.6,0.9,3.5,0.9,5.1-0.1c3.6-2.2,3.2-2,3.4-2.1c2.2-1.5,2.9-4.5,1.5-6.8L32,31.8
			c-1.4-2.3-4.5-3.1-6.8-1.6l-3.2,2c-1.5,0.9-2.4,2.5-2.4,4.2c0,0.2-0.1,0.3-0.3,0.2c-1.2-0.3-2.4-0.8-3.3-1.7
			c-1.4-1.3-2.6-2.6-3.7-4c-0.2-0.3-0.7-0.4-1-0.1s-0.4,0.7-0.1,1c1.1,1.5,2.4,2.9,3.9,4.2c1.1,1,2.5,1.7,3.9,2c1,0.2,2-0.6,2-1.6
			c0-1.2,0.6-2.4,1.7-3l2.7-1.7l6.3,10.2l-2.7,1.7C27.8,44.2,26.4,44.2,25.2,43.6L25.2,43.6z M26.6,31c1.5-0.5,3.3,0.1,4.1,1.5
			l2.5,4.1c0.9,1.4,0.6,3.2-0.5,4.4L26.6,31z"></path>
		<path d="M0.4,23.2c1,3.8,2.4,7.2,4.2,10.2s4.2,5.8,7.1,8.4c1.6,1.4,3.6,2.4,5.7,2.9c0.4,0.1,0.8-0.2,0.8-0.5
			c0.1-0.4-0.2-0.8-0.5-0.8c-1.9-0.4-3.6-1.3-5.1-2.5c-2.8-2.5-5.1-5.2-6.9-8.1c-1.7-2.8-3.1-6.1-4.1-9.8C0.9,19.4,1.7,15.7,4,13
			c0.6-0.7,1-1.7,0.9-2.7C4.8,9,5.5,7.8,6.6,7.1l2.7-1.7l6.3,10.2l-2.7,1.7c-1,0.6-2.3,0.7-3.4,0.2c-0.9-0.5-2,0.1-2.3,1.1
			c-0.4,1.5-0.4,3,0,4.5c0.6,2.1,1.3,4.1,2.2,5.8c0.2,0.3,0.6,0.5,1,0.3c0.3-0.2,0.5-0.6,0.3-1c-0.9-1.7-1.6-3.5-2.1-5.5
			c-0.3-1.2-0.3-2.5,0-3.7c0-0.1,0.2-0.2,0.3-0.2c1.5,0.8,3.3,0.7,4.8-0.2l3.2-2c2.3-1.4,3.1-4.5,1.6-6.8l-2.5-4.1
			c-1.4-2.4-4.5-3.1-6.8-1.6l-3.2,2c-1.5,0.9-2.5,2.7-2.4,4.5c0,0.6-0.2,1.2-0.6,1.7C0.3,15.2-0.6,19.3,0.4,23.2L0.4,23.2z
			 M14.7,6.3l2.5,4.1c0.9,1.4,0.6,3.2-0.5,4.4L10.6,4.8C12.1,4.3,13.9,4.9,14.7,6.3L14.7,6.3z"></path>
	</g>
</g>
</svg>',
				),
				'fi-payment-security-48' => array(
					'label' => esc_html__( 'Secure Payment', 'redparts-sputnik' ),
					'code'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 48 48" width="48" height="48">
<path d="M30,34.4H2.5c-0.5,0-0.9-0.4-0.9-0.9V7.7c0-0.5,0.4-0.9,0.9-0.9H42c0.5,0,0.9,0.4,0.9,0.9v11.2c0,0.4,0.4,0.8,0.8,0.8
	c0.4,0,0.8-0.4,0.8-0.8V7.7c0-1.4-1.1-2.5-2.5-2.5H2.5C1.1,5.2,0,6.3,0,7.7v25.8C0,34.8,1.1,36,2.5,36H30c0.4,0,0.8-0.4,0.8-0.8
	C30.8,34.7,30.5,34.4,30,34.4z"></path>
<path d="M15.4,18v-5.2c0-0.9-0.7-1.7-1.7-1.7H6.8c-0.9,0-1.7,0.7-1.7,1.7V18c0,0.9,0.7,1.7,1.7,1.7h6.9C14.6,19.7,15.4,18.9,15.4,18
	z M13.7,12.8V18c0,0,0,0.1-0.1,0.1h-3.5v-1.8h0.9c0.4,0,0.8-0.4,0.8-0.8c0-0.4-0.4-0.8-0.8-0.8h-0.9v-1.8L13.7,12.8
	C13.7,12.8,13.7,12.8,13.7,12.8z M6.8,18v-5.2c0,0,0-0.1,0.1-0.1h1.8V18L6.8,18C6.8,18,6.8,18,6.8,18z"></path>
<path d="M32.2,11.1c-0.8-0.5-1.7-0.8-2.6-0.8c-2.6,0-4.7,2.1-4.7,4.7s2.1,4.7,4.7,4.7c1,0,1.8-0.3,2.6-0.8c0.8,0.5,1.7,0.8,2.6,0.8
	c2.6,0,4.7-2.1,4.7-4.7s-2.1-4.7-4.7-4.7C33.8,10.3,32.9,10.6,32.2,11.1z M26.5,15c0-1.7,1.4-3.1,3.1-3.1c0.5,0,0.9,0.1,1.4,0.3
	C30.4,13,30.1,14,30.1,15c0,1,0.3,1.9,0.9,2.7c-0.4,0.2-0.9,0.3-1.4,0.3C27.9,18,26.5,16.7,26.5,15z M37.8,15c0,1.7-1.4,3.1-3.1,3.1
	c-0.5,0-0.9-0.1-1.4-0.3c0.6-0.8,0.9-1.7,0.9-2.7c0-0.4-0.4-0.8-0.8-0.8s-0.8,0.4-0.8,0.8c0,0.6-0.2,1.2-0.5,1.6
	c-0.3-0.5-0.5-1.1-0.5-1.6c0-1.7,1.4-3.1,3.1-3.1C36.4,11.9,37.8,13.3,37.8,15z"></path>
<path d="M6,24.1c-0.4,0-0.8,0.4-0.8,0.8c0,0.4,0.4,0.8,0.8,0.8h6.9c0.4,0,0.8-0.4,0.8-0.8c0-0.4-0.4-0.8-0.8-0.8H6z"></path>
<path d="M30.9,29.2H6c-0.4,0-0.8,0.4-0.8,0.8c0,0.4,0.4,0.8,0.8,0.8h24.9c0.4,0,0.8-0.4,0.8-0.8S31.3,29.2,30.9,29.2z"></path>
<path d="M16.3,24.1c-0.4,0-0.8,0.4-0.8,0.8c0,0.4,0.4,0.8,0.8,0.8h6.9c0.4,0,0.8-0.4,0.8-0.8c0-0.4-0.4-0.8-0.8-0.8H16.3z"></path>
<path d="M31.7,24.1h-5.2c-0.4,0-0.8,0.4-0.8,0.8c0,0.4,0.4,0.8,0.8,0.8h5.2c0.4,0,0.8-0.4,0.8-0.8C32.5,24.4,32.2,24.1,31.7,24.1z"
	></path>
<path d="M46.3,30.2v-3.6c0-3.3-2.7-6-6-6c-3.3,0-6,2.7-6,6v3.6c-1,0.3-1.7,1.3-1.7,2.4v7.7c0,1.4,1.1,2.5,2.5,2.5h10.3
	c1.4,0,2.5-1.1,2.5-2.5v-7.7C48,31.5,47.3,30.5,46.3,30.2z M40.3,22.2c2.4,0,4.3,2,4.3,4.3v3.5H36v-3.5C36,24.2,37.9,22.2,40.3,22.2
	z M46.4,40.3c0,0.5-0.4,0.9-0.9,0.9H35.2c-0.5,0-0.9-0.4-0.9-0.9v-7.7c0-0.5,0.4-0.9,0.9-0.9h10.3c0.5,0,0.9,0.4,0.9,0.9V40.3z"></path>
<path d="M40.3,33.5c-1.2,0-2.1,0.9-2.1,2.1c0,0.9,0.5,1.6,1.3,1.9v1.1c0,0.4,0.4,0.8,0.8,0.8s0.8-0.4,0.8-0.8v-1.1
	c0.8-0.3,1.3-1.1,1.3-1.9C42.4,34.4,41.5,33.5,40.3,33.5z M40.3,35.1c0.3,0,0.5,0.2,0.5,0.5s-0.2,0.5-0.5,0.5s-0.5-0.2-0.5-0.5
	S40.1,35.1,40.3,35.1z"></path>
</svg>',
				),
				'fi-tag-48'              => array(
					'label' => esc_html__( 'Tag Hot', 'redparts-sputnik' ),
					'code'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 48 48" width="48" height="48">
<path d="M48,3.3c0-0.9-0.3-1.7-1-2.3c-0.6-0.6-1.4-1-2.3-1c-0.9,0-1.7,0.3-2.3,1c-0.3,0.3-0.7,0.8-1,1.1c-0.3,0.3-0.2,0.8,0.1,1.1
	c0.3,0.3,0.8,0.2,1.1-0.1c0.4-0.5,0.7-0.9,0.9-1c0.3-0.3,0.8-0.5,1.2-0.5c0,0,0,0,0,0c0.5,0,0.9,0.2,1.2,0.5
	c0.3,0.3,0.5,0.8,0.5,1.2c0,0.5-0.2,0.9-0.5,1.2c-0.3,0.3-1.3,1.1-2.7,2.1c-0.9-1.5-2.4-2.4-4.3-2.4H27.5c-1.5,0-3,0.6-4.1,1.7
	L0.7,28.6C0.3,29,0,29.7,0,30.3s0.3,1.3,0.7,1.7L16,47.3c0.5,0.5,1.1,0.7,1.7,0.7c0.7,0,1.3-0.3,1.7-0.7l22.8-22.8
	c1.1-1.1,1.7-2.5,1.7-4.1V9.1c0-0.3,0-0.7-0.1-1C45.4,7,46.6,6,47,5.6C47.7,5,48,4.1,48,3.3z M42.3,9.1v11.4c0,1.1-0.4,2.2-1.2,3
	L18.3,46.2c-0.3,0.3-0.9,0.3-1.2,0L1.8,30.9c-0.3-0.3-0.3-0.9,0-1.2L24.6,6.9c0.8-0.8,1.8-1.2,3-1.2h11.4c1.3,0,2.4,0.7,3,1.8
	c-0.9,0.6-1.9,1.3-3,1.9c0,0-0.1-0.1-0.1-0.1c-1.3-1.3-3.3-1.3-4.6,0s-1.3,3.3,0,4.6c0.6,0.6,1.5,1,2.3,1c0.8,0,1.7-0.3,2.3-1
	c0.9-0.9,1.1-2.1,0.8-3.1C40.6,10.2,41.5,9.6,42.3,9.1C42.3,9.1,42.3,9.1,42.3,9.1z M35.7,11.9c0.1,0.3,0.4,0.4,0.7,0.4
	c0.1,0,0.2,0,0.3-0.1c0.5-0.2,0.9-0.5,1.4-0.7c0,0.4-0.2,0.9-0.5,1.2c-0.7,0.7-1.8,0.7-2.4,0c-0.7-0.7-0.7-1.8,0-2.4
	c0.3-0.3,0.8-0.5,1.2-0.5c0.3,0,0.7,0.1,1,0.3c-0.4,0.2-0.9,0.5-1.3,0.7C35.7,11.1,35.6,11.5,35.7,11.9z"></path>
<path d="M16.3,25.8c-0.3-0.3-0.8-0.3-1.1,0c-0.3,0.3-0.3,0.8,0,1.1l2.4,2.4l-3,3l-2.4-2.4c-0.3-0.3-0.8-0.3-1.1,0
	c-0.3,0.3-0.3,0.8,0,1.1l5.9,5.9c0.2,0.2,0.4,0.2,0.5,0.2s0.4-0.1,0.5-0.2c0.3-0.3,0.3-0.8,0-1.1l-2.4-2.4l3-3l2.4,2.4
	c0.2,0.2,0.4,0.2,0.5,0.2s0.4-0.1,0.5-0.2c0.3-0.3,0.3-0.8,0-1.1L16.3,25.8z"></path>
<path d="M24.8,21.4c-1.4-1.4-3.8-1.4-5.2,0s-1.4,3.8,0,5.2l1.8,1.8c0.7,0.7,1.7,1.1,2.6,1.1s1.9-0.4,2.6-1.1c1.4-1.4,1.4-3.8,0-5.2
	L24.8,21.4z M25.5,27.3c-0.8,0.8-2.2,0.8-3,0l-1.8-1.8c-0.8-0.8-0.8-2.2,0-3c0.4-0.4,1-0.6,1.5-0.6s1.1,0.2,1.5,0.6l1.8,1.8
	C26.3,25.1,26.3,26.5,25.5,27.3z"></path>
<path d="M27.4,15.8l1.8-1.8c0.3-0.3,0.3-0.8,0-1.1c-0.3-0.3-0.8-0.3-1.1,0l-4.7,4.7c-0.3,0.3-0.3,0.8,0,1.1c0.2,0.2,0.4,0.2,0.5,0.2
	s0.4-0.1,0.5-0.2l1.8-1.8l5.3,5.3c0.2,0.2,0.4,0.2,0.5,0.2c0.2,0,0.4-0.1,0.5-0.2c0.3-0.3,0.3-0.8,0-1.1L27.4,15.8z"></path>
</svg>',
				),
			);

			return apply_filters( 'redparts_sputnik_widget_block_features_icons', $icons );
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$layout   = ! empty( $instance['layout'] ) ? $instance['layout'] : 'classic';
			$features = ! empty( $instance['features'] ) ? array_values( $instance['features'] ) : array();

			if ( 'widget' === $layout ) {
				?>
				<!-- .th-block-features -->
				<div class="th-block th-block-features th-block-features--layout--<?php echo esc_attr( $layout ); ?>">
					<?php $this->features_list( $features ); ?>
				</div>
				<!-- .th-block-features / end -->
				<?php
			} else {
				?>
				<!-- .th-block-features -->
				<div class="th-block th-block-features th-block-features--layout--<?php echo esc_attr( $layout ); ?>">
					<div class="th-container">
						<?php $this->features_list( $features ); ?>
					</div>
				</div>
				<!-- .th-block-features / end -->
				<?php
			}
		}

		/**
		 * Outputs list of shop features.
		 *
		 * @param array $features List of shop features.
		 */
		protected function features_list( array $features ) {
			$icons = $this->icons();

			?>
			<ul class="th-block-features__list">
				<?php foreach ( $features as $feature ) : ?>
					<?php
					$feature_icon_type = 'predefined';
					$feature_icon      = 'fi-free-delivery-48';
					$feature_title     = '';
					$feature_subtitle  = '';

					if ( ! empty( $feature['icon_type'] ) ) {
						$feature_icon_type = $feature['icon_type'];
					}
					if ( 'predefined' === $feature_icon_type && ! empty( $feature['predefined_icon'] ) ) {
						$feature_icon = $feature['predefined_icon'];
					}
					if ( 'custom' === $feature_icon_type && ! empty( $feature['custom_icon'] ) ) {
						$feature_icon = $feature['custom_icon'];
					}

					if ( isset( $feature['title'] ) ) {
						$feature_title = $feature['title'];
					}
					if ( isset( $feature['subtitle'] ) ) {
						$feature_subtitle = $feature['subtitle'];
					}
					?>
					<li class="th-block-features__item">
						<div class="th-block-features__item-icon">
							<?php
							if ( 'predefined' === $feature_icon_type && ! empty( $icons[ $feature_icon ] ) ) :
								echo wp_kses( $icons[ $feature_icon ]['code'], 'redparts_sputnik_svg' );
							elseif ( 'custom' === $feature_icon_type && wp_attachment_is_image( $feature_icon ) ) :
								echo wp_get_attachment_image( $feature_icon );
							endif;
							?>
						</div>
						<div class="th-block-features__item-info">
							<div class="th-block-features__item-title">
								<?php echo esc_html( $feature_title ); ?>
							</div>
							<div class="th-block-features__item-subtitle">
								<?php echo esc_html( $feature_subtitle ); ?>
							</div>
						</div>
					</li>
					<li class="th-block-features__divider" role="presentation"></li>
				<?php endforeach; ?>
			</ul>
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			$icon_options = array();

			foreach ( $this->icons() as $icon_key => $icon_data ) {
				$icon_options[] = array(
					'key'   => $icon_key,
					'label' => $icon_data['label'],
				);
			}

			return array(
				array(
					'key'     => 'layout',
					'label'   => esc_html__( 'Layout', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'top-strip',
							'label' => esc_html__( 'Top Strip', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'bottom-strip',
							'label' => esc_html__( 'Bottom Strip', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'widget',
							'label' => esc_html__( 'Widget', 'redparts-sputnik' ),
						),
					),
				),
				array(
					'key'    => 'features',
					'label'  => esc_html__( 'Features', 'redparts-sputnik' ),
					'type'   => 'tabs',
					'max'    => 4,
					'fields' => array(
						array(
							'key'   => 'priority',
							'label' => esc_html__( 'Priority', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'subtitle',
							'label'        => esc_html__( 'Subtitle', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'     => 'icon_type',
							'label'   => esc_html__( 'Icon type', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'predefined',
							'options' => array(
								array(
									'key'   => 'predefined',
									'label' => esc_html__( 'Predefined', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'custom',
									'label' => esc_html__( 'Custom', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'      => 'predefined_icon',
							'label'    => esc_html__( 'Predefined icon', 'redparts-sputnik' ),
							'type'     => 'select',
							'options'  => $icon_options,
							'required' => array( array( 'icon_type', '=', 'predefined' ) ),
						),
						array(
							'key'      => 'custom_icon',
							'label'    => esc_html__( 'Custom icon', 'redparts-sputnik' ),
							'type'     => 'image',
							'required' => array( array( 'icon_type', '=', 'custom' ) ),
						),
					),
					'l10n'   => array(
						'add'    => esc_html__( 'Add Feature', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Feature', 'redparts-sputnik' ),
						/* translators: %s: Feature number. */
						'title'  => esc_html__( 'Feature %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );
		}
	}
}
