<?php
/**
 * Block posts carousel widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Posts_Carousel' ) ) {
	/**
	 * Class Block_Posts_Carousel
	 */
	class Block_Posts_Carousel extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_posts_carousel',
				'description'                 => esc_html_x( 'Posts carousel.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_posts_carousel',
				esc_html_x( 'RedParts: Block Posts Carousel', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$title                = ! empty( $instance['title'] ) ? $instance['title'] : '';
			$layout               = ! empty( $instance['layout'] ) ? $instance['layout'] : 'grid';
			$links                = isset( $instance['links'] ) && is_array( $instance['links'] ) ? array_values( $instance['links'] ) : array();
			$autoplay             = ( $instance['autoplay'] ?? 'no' ) === 'yes';
			$autoplay_hover_pause = ( $instance['autoplay_hover_pause'] ?? 'yes' ) === 'yes';
			$autoplay_timeout     = absint( $instance['autoplay_timeout'] ?? '5000' );

			$header_links = array();

			foreach ( $links as $header_link ) {
				$header_link = is_array( $header_link ) ? $header_link : array();

				$header_links[] = array(
					'url'   => isset( $header_link['url'] ) ? $header_link['url'] : '',
					'title' => isset( $header_link['title'] ) ? $header_link['title'] : '',
					'order' => isset( $header_link['order'] ) ? $header_link['order'] : 0,
				);
			}

			uasort(
				$header_links,
				function( $a, $b ) {
					$a = intval( $a['order'] );
					$b = intval( $b['order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			$query = new WP_Query(
				array(
					'post_type'      => 'post',
					'posts_per_page' => 8,
					'order'          => 'DESC',
					'orderby'        => 'date',
				)
			);

			$root_classes = array(
				'th-block',
				'th-block-posts-carousel',
				'th-block-posts-carousel--layout--' . $layout,
			);

			?>
			<!-- .th-block-posts-carousel -->
			<div
				class="<?php echo esc_attr( implode( ' ', $root_classes ) ); ?>"
				data-layout="<?php echo esc_attr( $layout ); ?>"
				data-autoplay="<?php echo esc_attr( $autoplay ? 1 : 0 ); ?>"
				data-autoplay-hover-pause="<?php echo esc_attr( $autoplay_hover_pause ? 1 : 0 ); ?>"
				data-autoplay-timeout="<?php echo esc_attr( $autoplay_timeout ); ?>"
			>
				<div class="th-container">
					<div class="th-section-header">
						<div class="th-section-header__body">
							<?php if ( ! empty( $title ) ) : ?>
								<h2 class="th-section-header__title">
									<?php echo esc_html( $title ); ?>
								</h2>
							<?php endif; ?>

							<div class="th-section-header__spring"></div>

							<?php if ( ! empty( $header_links ) && empty( $header_groups ) ) : ?>
								<ul class="th-section-header__links">
									<?php foreach ( $header_links as $link ) : ?>
										<li class="th-section-header__links-item">
											<a
												class="th-section-header__links-link"
												href="<?php echo esc_attr( $link['url'] ); ?>"
											>
												<?php echo esc_html( $link['title'] ); ?>
											</a>
										</li>
									<?php endforeach; ?>
								</ul>
							<?php endif; ?>

							<div class="th-section-header__arrows">
								<div
									class="th-arrow th-arrow--direction--prev th-section-header__arrow th-section-header__arrow--prev">
									<button class="th-arrow__button" type="button">
										<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
											<path
												d="M6.7.3c-.4-.4-.9-.4-1.3 0L0 5.5l5.4 5.2c.4.4.9.3 1.3 0 .4-.4.4-1 0-1.3l-4-3.9 4-3.9c.4-.4.4-1 0-1.3z"></path>
										</svg>
									</button>
								</div>
								<div
									class="th-arrow th-arrow--direction--next th-section-header__arrow th-section-header__arrow--next">
									<button class="th-arrow__button" type="button">
										<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
											<path
												d="M.3 10.7c.4.4.9.4 1.3 0L7 5.5 1.6.3C1.2-.1.7 0 .3.3c-.4.4-.4 1 0 1.3l4 3.9-4 3.9c-.4.4-.4 1 0 1.3z"></path>
										</svg>
									</button>
								</div>
							</div>

							<div class="th-section-header__divider"></div>
						</div>
					</div>

					<div class="th-block-posts-carousel__carousel">
						<div class="owl-carousel">
							<?php while ( $query->have_posts() ) : ?>
								<?php $query->the_post(); ?>
								<div class="th-block-posts-carousel__item">
									<div id="post-<?php the_ID(); ?>" <?php post_class( 'th-post-card' ); ?>>
										<?php if ( ! post_password_required() && ! is_attachment() && has_post_thumbnail() ) : ?>
											<div class="th-post-card__image">
												<a href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
													<?php
													the_post_thumbnail(
														'redparts-post-card-thumbnail',
														array(
															'alt' => the_title_attribute( array( 'echo' => false ) ),
														)
													);
													?>
												</a>
											</div>
										<?php endif; ?>
										<div class="th-post-card__content">
											<?php
											if ( 'post' === get_post_type() ) {
												/* translators: used between list items, there is a space after the comma */
												$categories_list = get_the_category_list( esc_html__( ', ', 'redparts-sputnik' ) );

												if ( $categories_list ) {
													?>
													<div class="th-post-card__category">
														<?php echo wp_kses( $categories_list, 'redparts_sputnik_categories_list' ); ?>
													</div>
													<?php
												}
											}
											?>
											<div class="th-post-card__title">
												<h2>
													<a href="<?php echo esc_url( get_permalink() ); ?>" rel="bookmark">
														<?php the_title(); ?>
													</a>
												</h2>
											</div>
											<div class="th-post-card__meta">
												<div class="th-post-card__meta-list">
													<?php $author_id = get_the_author_meta( 'ID' ); ?>

													<a
														href="<?php echo esc_url( get_author_posts_url( $author_id ) ); ?>"
														class="th-post-card__meta-item th-post-card__meta-item--author"
													>
														<div class="th-post-card__meta-icon">
															<?php echo get_avatar( $author_id, 18 ); ?>
														</div>
														<div class="th-post-card__meta-value">
															<?php echo esc_html( get_the_author() ); ?>
														</div>
													</a>

													<a
														href="<?php echo esc_url( get_permalink() ); ?>"
														class="th-post-card__meta-item th-post-card__meta-item--date"
													>
														<div class="th-post-card__meta-icon">
															<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16">
																<path d="M15,14.5c0,0.8-0.7,1.5-1.5,1.5h-11C1.7,16,1,15.3,1,14.5V6h14V14.5z M5,8.5C5,8.2,4.8,8,4.5,8h-1C3.2,8,3,8.2,3,8.5v1C3,9.8,3.2,10,3.5,10h1C4.8,10,5,9.8,5,9.5V8.5z M5,12.5C5,12.2,4.8,12,4.5,12h-1C3.2,12,3,12.2,3,12.5v1C3,13.8,3.2,14,3.5,14h1C4.8,14,5,13.8,5,13.5V12.5z M9,8.5C9,8.2,8.8,8,8.5,8h-1C7.2,8,7,8.2,7,8.5v1C7,9.8,7.2,10,7.5,10h1C8.8,10,9,9.8,9,9.5V8.5zM9,12.5C9,12.2,8.8,12,8.5,12h-1C7.2,12,7,12.2,7,12.5v1C7,13.8,7.2,14,7.5,14h1C8.8,14,9,13.8,9,13.5V12.5z M13,8.5C13,8.2,12.8,8,12.5,8h-1C11.2,8,11,8.2,11,8.5v1c0,0.3,0.2,0.5,0.5,0.5h1c0.3,0,0.5-0.2,0.5-0.5V8.5z M13,12.5c0-0.3-0.2-0.5-0.5-0.5h-1c-0.3,0-0.5,0.2-0.5,0.5v1c0,0.3,0.2,0.5,0.5,0.5h1c0.3,0,0.5-0.2,0.5-0.5V12.5z M1,3.5C1,2.7,1.7,2,2.5,2H4V0.5C4,0.2,4.2,0,4.5,0h1C5.8,0,6,0.2,6,0.5V2h4V0.5C10,0.2,10.2,0,10.5,0h1C11.8,0,12,0.2,12,0.5V2h1.5C14.3,2,15,2.7,15,3.5V5H1V3.5z"/>
															</svg>
														</div>
														<time
															class="th-post-card__meta-value published updated"
															datetime="<?php echo esc_attr( get_the_date( DATE_W3C ) ); ?>"
														>
															<?php echo esc_html( get_the_date() ); ?>
														</time>
													</a>

													<a
														href="<?php echo esc_url( get_comments_link() ); ?>"
														class="th-post-card__meta-item th-post-card__meta-item--comments"
													>
														<div class="th-post-card__meta-icon">
															<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16">
																<path d="M16,7.9c-0.1,3.6-5.4,7.5-11.4,5.5c-1,0.8-2.4,1.5-4.2,1.6c-0.3,0-0.6-0.2-0.3-0.4c0.6-0.6,1.4-1.8,1.7-3C-0.9,9.2-1.1,3.5,5.4,1.4C9.1,0.1,16.2,1.9,16,7.9z"/>
															</svg>
														</div>
														<div class="th-post-card__meta-value">
															<?php
															echo esc_html(
																sprintf(
																	/* translators: %s Comments count. */
																	_n( '%s comment', '%s comments', get_comments_number(), 'redparts-sputnik' ), // SKIP-ESC.
																	get_comments_number()
																)
															);
															?>
														</div>
													</a>
												</div>
											</div>
											<?php if ( '' !== get_post()->post_content ) : ?>
												<div class="th-post-card__excerpt">
													<?php the_excerpt(); ?>
												</div>
											<?php endif; ?>
											<div class="th-post-card__more">
												<a
													href="<?php echo esc_url( get_permalink() ); ?>"
													rel="bookmark"
													class="th-button th-button--style--primary th-button--size--small"
												>
													<?php esc_html_e( 'Read More', 'redparts-sputnik' ); ?>

													<svg xmlns="http://www.w3.org/2000/svg" width="6px" height="9px">
														<path d="M0.3,7.4l3-2.9l-3-2.9c-0.4-0.3-0.4-0.9,0-1.3l0,0c0.4-0.3,0.9-0.4,1.3,0L6,4.5L1.6,8.7c-0.4,0.4-0.9,0.4-1.3,0l0,0C-0.1,8.4-0.1,7.8,0.3,7.4z"/>
													</svg>
												</a>
											</div>
										</div>
									</div>
								</div>
							<?php endwhile; ?>
						</div>
					</div>
				</div>
			</div>
			<!-- .th-block-posts-carousel / end -->
			<?php

			$query->reset_postdata();
		}

		/**
		 * Returns form fields.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'     => 'layout',
					'label'   => esc_html__( 'Layout', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'grid',
							'label' => esc_html__( 'Grid', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'list',
							'label' => esc_html__( 'List', 'redparts-sputnik' ),
						),
					),
				),
				array(
					'key'     => 'autoplay',
					'label'   => esc_html__( 'Autoplay', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'no',
				),
				array(
					'key'     => 'autoplay_hover_pause',
					'label'   => esc_html__( 'Pause autoplay on hover', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'yes',
				),
				array(
					'key'     => 'autoplay_timeout',
					'label'   => esc_html__( 'Autoplay timeout', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => '5000',
				),
				array(
					'key'        => 'links',
					'select_key' => 'current_link',
					'label'      => esc_html__( 'Links', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'fields'     => array(
						array(
							'key'          => 'url',
							'label'        => esc_html__( 'URL', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Link', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Link', 'redparts-sputnik' ),
						/* translators: %s: Group number. */
						'title'  => esc_html__( 'Link %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-posts-carousel' );
		}
	}
}
