<?php
/**
 * Block products carousel widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Products_Carousel' ) ) {
	/**
	 * Class Block_Products_Carousel
	 */
	class Block_Products_Carousel extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_products_carousel',
				'description'                 => esc_html_x( 'Products carousel.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_products_carousel',
				esc_html_x( 'RedParts: Block Products Carousel', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			$title                = isset( $instance['title'] ) ? $instance['title'] : '';
			$layout               = isset( $instance['layout'] ) ? $instance['layout'] : 'grid-4';
			$mobile_grid_columns  = max( 1, min( 2, abs( $instance['mobile_grid_columns'] ?? '1' ) ) );
			$rows                 = max( 1, intval( isset( $instance['rows'] ) ? $instance['rows'] : 1 ) );
			$links                = isset( $instance['links'] ) && is_array( $instance['links'] ) ? array_values( $instance['links'] ) : array();
			$groups               = isset( $instance['groups'] ) && is_array( $instance['groups'] ) ? array_values( $instance['groups'] ) : array();
			$autoplay             = ( $instance['autoplay'] ?? 'no' ) === 'yes';
			$autoplay_hover_pause = ( $instance['autoplay_hover_pause'] ?? 'yes' ) === 'yes';
			$autoplay_timeout     = absint( $instance['autoplay_timeout'] ?? '5000' );

			$header_links = array();

			foreach ( $links as $header_link ) {
				$header_link = is_array( $header_link ) ? $header_link : array();

				$header_links[] = array(
					'url'   => isset( $header_link['url'] ) ? $header_link['url'] : '',
					'title' => isset( $header_link['title'] ) ? $header_link['title'] : '',
					'order' => isset( $header_link['order'] ) ? $header_link['order'] : 0,
				);
			}

			uasort(
				$header_links,
				function( $a, $b ) {
					$a = intval( $a['order'] );
					$b = intval( $b['order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			$current_group = 0;

			if ( isset( $instance['current_group'] ) && ( $this->is_preview() || $this->is_elementor_editor() || wp_doing_ajax() ) ) {
				$current_group = absint( $instance['current_group'] );
				$current_group = isset( $groups[ $current_group ] ) ? $current_group : 0;
			}

			$header_groups = array();

			foreach ( $groups as $index => $header_group ) {
				$header_group = is_array( $header_group ) ? $header_group : array();

				$header_groups[] = array(
					'name'        => isset( $header_group['title'] ) ? $header_group['title'] : '',
					'active'      => $current_group === $index,
					'group_order' => isset( $header_group['group_order'] ) ? $header_group['group_order'] : 0,
					'data'        => array(
						'widget_id'   => $this->number,
						'group_index' => $index,
					),
				);
			}

			uasort(
				$header_groups,
				function( $a, $b ) {
					$a = intval( $a['group_order'] );
					$b = intval( $b['group_order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			$number     = 0;
			$show       = 'all';
			$products   = '';
			$categories = '';
			$orderby    = 'date';
			$order      = 'desc';

			if ( count( $groups ) && isset( $groups[ $current_group ] ) ) {
				$group      = $groups[ $current_group ];
				$number     = isset( $group['number'] ) ? absint( $group['number'] ) : $number;
				$show       = isset( $group['show'] ) ? sanitize_title( $group['show'] ) : $show;
				$products   = isset( $group['products'] ) ? wp_parse_slug_list( $group['products'] ) : $products;
				$categories = isset( $group['categories'] ) ? wp_parse_slug_list( $group['categories'] ) : $categories;
				$orderby    = isset( $group['orderby'] ) ? sanitize_title( $group['orderby'] ) : $orderby;
				$order      = isset( $group['order'] ) ? sanitize_title( $group['order'] ) : $order;
			}

			$product_visibility_term_ids = wc_get_product_visibility_term_ids();

			$query_args = array(
				'posts_per_page' => $number,
				'post_status'    => 'publish',
				'post_type'      => 'product',
				'no_found_rows'  => 1,
				'order'          => $order,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
				'tax_query'      => array(
					'relation' => 'AND',
				),
			);

			if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
				$query_args['tax_query'][] = array(
					array(
						'taxonomy' => 'product_visibility',
						'field'    => 'term_taxonomy_id',
						'terms'    => $product_visibility_term_ids['outofstock'],
						'operator' => 'NOT IN',
					),
				);
			}

			if ( ! empty( $products ) ) {
				$query_args['post_name__in'] = $products;
			}

			if ( ! empty( $categories ) ) {
				$query_args['tax_query'][] = array(
					'taxonomy' => 'product_cat',
					'field'    => 'slug',
					'terms'    => $categories,
				);
			}

			switch ( $show ) {
				case 'featured':
					$query_args['tax_query'][] = array(
						'taxonomy' => 'product_visibility',
						'field'    => 'term_taxonomy_id',
						'terms'    => $product_visibility_term_ids['featured'],
					);
					break;
				case 'onsale':
					$product_ids_on_sale    = wc_get_product_ids_on_sale();
					$product_ids_on_sale[]  = 0;
					$query_args['post__in'] = $product_ids_on_sale;
					break;
			}

			switch ( $orderby ) {
				case 'price':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = '_price';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'sales':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = 'total_sales';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'rating':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = '_wc_average_rating';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'product_slugs':
					$query_args['orderby'] = 'post_name__in';
					break;
				default:
					$query_args['orderby'] = 'date';
			}

			$products = new WP_Query( $query_args );

			$nonce = wp_create_nonce( 'redparts_sputnik_widget' );

			$product_card_layout = array(
				'grid-4'     => 'grid',
				'grid-5'     => 'grid',
				'grid-6'     => 'grid',
				'horizontal' => 'horizontal',
			)[ $layout ];

			?>
			<!-- .th-block-products-carousel -->
			<div
				class="th-block th-block-products-carousel"
				data-layout="<?php echo esc_attr( $layout ); ?>"
				data-ajax-url="<?php echo esc_url( apply_filters( 'redparts_sputnik_ajax_url', '' ) ); ?>"
				data-nonce="<?php echo esc_attr( $nonce ); ?>"
				data-autoplay="<?php echo esc_attr( $autoplay ? 1 : 0 ); ?>"
				data-autoplay-hover-pause="<?php echo esc_attr( $autoplay_hover_pause ? 1 : 0 ); ?>"
				data-autoplay-timeout="<?php echo esc_attr( $autoplay_timeout ); ?>"
				data-mobile-grid-columns="<?php echo esc_attr( $mobile_grid_columns ); ?>"
			>
				<div class="th-container">
					<div class="th-section-header">
						<div class="th-section-header__body">
							<?php if ( ! empty( $title ) ) : ?>
								<h2 class="th-section-header__title">
									<?php echo esc_html( $title ); ?>
								</h2>
							<?php endif; ?>

							<div class="th-section-header__spring"></div>

							<?php if ( ! empty( $header_links ) && ( empty( $header_groups ) || 1 === count( $header_groups ) ) ) : ?>
								<ul class="th-section-header__links">
									<?php foreach ( $header_links as $link ) : ?>
										<li class="th-section-header__links-item">
											<a
												class="th-section-header__links-link"
												href="<?php echo esc_attr( $link['url'] ); ?>"
											>
												<?php echo esc_html( $link['title'] ); ?>
											</a>
										</li>
									<?php endforeach; ?>
								</ul>
							<?php endif; ?>

							<?php if ( ! empty( $header_groups ) && 1 < count( $header_groups ) ) : ?>
								<ul class="th-section-header__groups">
									<?php foreach ( $header_groups as $header_group ) : ?>
										<?php
										$button_classes = array( 'th-section-header__groups-button' );

										if ( $header_group['active'] ) {
											$button_classes[] = 'th-section-header__groups-button--active';
										}
										?>
										<li class="th-section-header__groups-item">
											<button
												type="button"
												class="<?php echo esc_attr( implode( ' ', $button_classes ) ); ?>"
												data-group-data="<?php echo esc_attr( wp_json_encode( $header_group['data'] ) ); ?>"
											>
												<?php echo esc_html( $header_group['name'] ); ?>
											</button>
										</li>
									<?php endforeach; ?>
								</ul>
							<?php endif; ?>

							<div class="th-section-header__arrows">
								<div
									class="th-arrow th-arrow--direction--prev th-section-header__arrow th-section-header__arrow--prev">
									<button class="th-arrow__button" type="button">
										<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
											<path
												d="M6.7.3c-.4-.4-.9-.4-1.3 0L0 5.5l5.4 5.2c.4.4.9.3 1.3 0 .4-.4.4-1 0-1.3l-4-3.9 4-3.9c.4-.4.4-1 0-1.3z"></path>
										</svg>
									</button>
								</div>
								<div
									class="th-arrow th-arrow--direction--next th-section-header__arrow th-section-header__arrow--next">
									<button class="th-arrow__button" type="button">
										<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
											<path
												d="M.3 10.7c.4.4.9.4 1.3 0L7 5.5 1.6.3C1.2-.1.7 0 .3.3c-.4.4-.4 1 0 1.3l4 3.9-4 3.9c-.4.4-.4 1 0 1.3z"></path>
										</svg>
									</button>
								</div>
							</div>

							<div class="th-section-header__divider"></div>
						</div>
					</div>

					<div class="th-block-products-carousel__carousel">
						<div class="th-block-products-carousel__carousel-loader"></div>
						<div class="owl-carousel">
							<?php while ( $products->have_posts() ) : ?>
								<div class="th-block-products-carousel__column">
									<?php for ( $i = 0; $i < $rows; $i++ ) : ?>
										<?php
										if ( $products->current_post + 1 >= $products->post_count ) {
											break;
										}
										?>
										<ul class="th-block-products-carousel__cell">
											<?php
											$products->the_post();

											if ( 'horizontal' === $product_card_layout ) {
												$exclude_elements = array(
													'featured_attributes',
													'description',
													'addtocart',
													'footer_wishlist',
													'footer_compare',
													'action_wishlist',
													'action_compare',
													'meta',
													'compatibility',
												);
											} else {
												$exclude_elements = array(
													'featured_attributes',
													'description',
													'footer_wishlist',
													'footer_compare',
												);
											}

											wc_set_loop_prop( 'redparts_class', 'th-block-products-carousel__item' );
											wc_set_loop_prop( 'redparts_product_card_class', 'th-product-card--layout--' . $product_card_layout );
											wc_set_loop_prop( 'redparts_product_card_exclude', $exclude_elements );

											wc_get_template_part( 'content', 'product' );
											?>
										</ul>
									<?php endfor; ?>
								</div>
							<?php endwhile; ?>
						</div>
					</div>
				</div>
			</div>
			<!-- .th-block-products-carousel / end -->
			<?php

			wp_reset_postdata();
			wc_reset_loop();
		}

		/**
		 * Returns form fields.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'     => 'layout',
					'label'   => esc_html__( 'Layout', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'grid-4',
							'label' => esc_html__( 'Grid 4', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'grid-5',
							'label' => esc_html__( 'Grid 5', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'grid-6',
							'label' => esc_html__( 'Grid 6', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'horizontal',
							'label' => esc_html__( 'List', 'redparts-sputnik' ),
						),
					),
				),
				array(
					'key'     => 'mobile_grid_columns',
					'label'   => esc_html__( 'Mobile grid columns', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => '1',
							'label' => esc_html__( '1 column', 'redparts-sputnik' ),
						),
						array(
							'key'   => '2',
							'label' => esc_html__( '2 columns', 'redparts-sputnik' ),
						),
					),
					'default' => '1',
				),
				array(
					'key'     => 'autoplay',
					'label'   => esc_html__( 'Autoplay', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'no',
				),
				array(
					'key'     => 'autoplay_hover_pause',
					'label'   => esc_html__( 'Pause autoplay on hover', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'yes',
				),
				array(
					'key'     => 'autoplay_timeout',
					'label'   => esc_html__( 'Autoplay timeout', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => '5000',
				),
				array(
					'key'   => 'rows',
					'label' => esc_html__( 'Rows', 'redparts-sputnik' ),
					'type'  => 'number',
				),
				array(
					'key'        => 'groups',
					'select_key' => 'current_group',
					'label'      => esc_html__( 'Products groups', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'fields'     => array(
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'     => 'group_order',
							'label'   => esc_html__( 'Group order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
						array(
							'key'     => 'number',
							'label'   => esc_html__( 'Number of products to show', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
						array(
							'key'     => 'show',
							'label'   => esc_html__( 'Show', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'all',
							'options' => array(
								array(
									'key'   => 'all',
									'label' => esc_html__( 'All products', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'featured',
									'label' => esc_html__( 'Featured products', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'onsale',
									'label' => esc_html__( 'On-sale products', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'   => 'products',
							'label' => esc_html__( 'Products (comma separated slugs)', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'categories',
							'label' => esc_html__( 'Categories (comma separated slugs)', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'     => 'orderby',
							'label'   => esc_html__( 'Order by', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'date',
							'options' => array(
								array(
									'key'   => 'date',
									'label' => esc_html__( 'Date', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'price',
									'label' => esc_html__( 'Price', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'sales',
									'label' => esc_html__( 'Sales', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'rating',
									'label' => esc_html__( 'Average rating', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'product_slugs',
									'label' => esc_html__( 'Product slugs', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'desc',
							'options' => array(
								array(
									'key'   => 'desc',
									'label' => esc_html__( 'DESC', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'asc',
									'label' => esc_html__( 'ASC', 'redparts-sputnik' ),
								),
							),
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Group', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Group', 'redparts-sputnik' ),
						/* translators: %s: Group number. */
						'title'  => esc_html__( 'Group %s', 'redparts-sputnik' ),
					),
				),
				array(
					'key'        => 'links',
					'select_key' => 'current_link',
					'label'      => esc_html__( 'Links', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'fields'     => array(
						array(
							'key'          => 'url',
							'label'        => esc_html__( 'URL', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Link', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Link', 'redparts-sputnik' ),
						/* translators: %s: Group number. */
						'title'  => esc_html__( 'Link %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-products-carousel' );
		}
	}
}
