<?php
/**
 * Block teammates widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Teammates' ) ) {
	/**
	 * Class Block_Teammates
	 */
	class Block_Teammates extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_teammates',
				'description'                 => esc_html_x( 'Displays teammates carousel.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_teammates',
				esc_html_x( 'RedParts: Block Teammates', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$title         = ! empty( $instance['title'] ) ? $instance['title'] : '';
			$subtitle      = ! empty( $instance['subtitle'] ) ? $instance['subtitle'] : '';
			$teammates_raw = isset( $instance['teammates'] ) && is_array( $instance['teammates'] ) ? array_values( $instance['teammates'] ) : array();
			$teammates     = array();

			foreach ( $teammates_raw as $teammate_raw ) {
				$teammate_raw = is_array( $teammate_raw ) ? $teammate_raw : array();
				$contacts     = array();

				foreach ( array( 1, 2, 3, 4, 5 ) as $contact_index ) {
					$contact_key  = 'contact_' . $contact_index;
					$contact_icon = isset( $teammate_raw[ $contact_key . '_icon' ] ) ? $teammate_raw[ $contact_key . '_icon' ] : '';
					$contact_url  = isset( $teammate_raw[ $contact_key . '_url' ] ) ? $teammate_raw[ $contact_key . '_url' ] : '';

					if ( ! empty( $contact_icon ) ) {
						$contacts[] = array(
							'icon' => $contact_icon,
							'url'  => $contact_url,
						);
					}
				}

				$teammates[] = array(
					'name'     => isset( $teammate_raw['name'] ) ? $teammate_raw['name'] : '',
					'bio'      => isset( $teammate_raw['bio'] ) ? $teammate_raw['bio'] : '',
					'position' => isset( $teammate_raw['position'] ) ? $teammate_raw['position'] : '',
					'contacts' => $contacts,
					'avatar'   => isset( $teammate_raw['avatar'] ) ? absint( $teammate_raw['avatar'] ) : '',
					'order'    => isset( $teammate_raw['order'] ) ? $teammate_raw['order'] : 0,
				);
			}

			uasort(
				$teammates,
				function( $a, $b ) {
					$a = intval( $a['order'] );
					$b = intval( $b['order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			?>
			<!-- .th-block-teammates -->
			<div class="th-block th-block-teammates">
				<div class="th-container th-container--max--xl">
					<?php if ( ! empty( $title ) || ! empty( $subtitle ) ) : ?>
						<div class="th-block-teammates__header">
							<?php if ( ! empty( $title ) ) : ?>
								<div class="th-block-teammates__title">
									<?php echo wp_kses( $title, 'redparts_sputnik_block_teammates_title' ); ?>
								</div>
							<?php endif; ?>
							<?php if ( ! empty( $subtitle ) ) : ?>
								<div class="th-block-teammates__subtitle">
									<?php echo wp_kses( $subtitle, 'redparts_sputnik_block_teammates_subtitle' ); ?>
								</div>
							<?php endif; ?>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $teammates ) ) : ?>
						<div class="th-block-teammates__list">
							<div class="owl-carousel">
								<?php foreach ( $teammates as $teammate ) : ?>
									<div class="th-block-teammates__item th-teammate">
										<?php if ( wp_attachment_is_image( $teammate['avatar'] ) ) : ?>
											<div class="th-teammate__avatar">
												<?php echo wp_get_attachment_image( $teammate['avatar'], array( 300, 300 ) ); ?>

												<div class="th-teammate__position">
													<?php echo esc_html( $teammate['position'] ); ?>
												</div>
											</div>
										<?php endif; ?>
										<div class="th-teammate__info">
											<div class="th-teammate__name">
												<?php echo esc_html( $teammate['name'] ); ?>
											</div>
											<?php if ( ! empty( $teammate['bio'] ) ) : ?>
												<div class="th-teammate__description">
													<?php echo esc_html( $teammate['bio'] ); ?>
												</div>
											<?php endif; ?>
											<?php if ( ! empty( $teammate['contacts'] ) ) : ?>
												<div class="th-teammate__contacts">
													<ul>
														<?php foreach ( $teammate['contacts'] as $contact ) : ?>
															<li>
																<a href="<?php echo esc_url( $contact['url'] ); ?>">
																	<i class="<?php echo esc_attr( $contact['icon'] ); ?>"></i>
																</a>
															</li>
														<?php endforeach; ?>
													</ul>
												</div>
											<?php endif; ?>
										</div>
									</div>
								<?php endforeach; ?>
							</div>
						</div>
					<?php endif; ?>
				</div>
			</div>
			<!-- .th-block-teammates / end -->
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'          => 'subtitle',
					'label'        => esc_html__( 'Subtitle', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => true,
				),
				array(
					'key'        => 'teammates',
					'select_key' => 'current_teammate',
					'label'      => esc_html__( 'Teammates', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'fields'     => array(
						array(
							'key'          => 'name',
							'label'        => esc_html__( 'Name', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'bio',
							'label'        => esc_html__( 'Short Biography', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'position',
							'label'        => esc_html__( 'Position', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),

						array(
							'key'   => 'contact_1_icon',
							'label' => esc_html__( 'Contact 1: Icon', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'contact_1_url',
							'label' => esc_html__( 'Contact 1: URL', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'contact_2_icon',
							'label' => esc_html__( 'Contact 2: Icon', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'contact_2_url',
							'label' => esc_html__( 'Contact 2: URL', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'contact_3_icon',
							'label' => esc_html__( 'Contact 3: Icon', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'contact_3_url',
							'label' => esc_html__( 'Contact 3: URL', 'redparts-sputnik' ),
							'type'  => 'text',
						),

						array(
							'key'   => 'avatar',
							'label' => esc_html__( 'Avatar', 'redparts-sputnik' ),
							'type'  => 'image',
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Teammate', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Teammate', 'redparts-sputnik' ),
						/* translators: %s: Teammate number. */
						'title'  => esc_html__( 'Teammate %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-teammates' );
		}
	}
}
