<?php
/**
 * Footer contacts widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Footer_Contacts' ) ) {
	/**
	 * Class Footer_Contacts
	 */
	class Footer_Contacts extends Widget {
		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_footer_contacts',
				'description'                 => esc_html_x( 'Displays contacts in the footer.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_footer_contacts',
				esc_html_x( 'RedParts: Footer Contacts', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Returns widget title.
		 *
		 * @param array $instance The settings for the particular instance of the widget.
		 *
		 * @return string
		 */
		public function get_title( array $instance ): string {
			$show_logo = ! empty( $instance['show_logo'] ) ? 'yes' === $instance['show_logo'] : false;

			if ( $show_logo && has_custom_logo() ) {
				return '';
			}

			return parent::get_title( $instance );
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$text      = ! empty( $instance['text'] ) ? $instance['text'] : '';
			$contacts  = ! empty( $instance['contacts'] ) ? $instance['contacts'] : array();
			$show_logo = ! empty( $instance['show_logo'] ) ? 'yes' === $instance['show_logo'] : false;
			$logo      = isset( $instance['logo'] ) ? absint( $instance['logo'] ) : '';
			$has_logo  = wp_get_attachment_image_url( $logo, 'full' );

			?>
			<div class="th-footer-contacts">
				<?php if ( $show_logo && ( has_custom_logo() || $has_logo ) ) : ?>
					<div class="th-footer-contacts__logo">
						<?php if ( $has_logo ) : ?>
							<a href="<?php echo esc_url( home_url( '/' ) ); ?>">
								<?php echo wp_get_attachment_image( $logo, 'full' ); ?>
							</a>
						<?php else : ?>
							<?php the_custom_logo(); ?>
						<?php endif; ?>
					</div>
				<?php endif; ?>

				<div class="th-footer-contacts__text"><?php echo wp_kses( $text, 'redparts_sputnik_text' ); ?></div>

				<?php if ( ! empty( $contacts ) ) : ?>
					<?php
					uasort(
						$contacts,
						function( $a, $b ) {
							$a = intval( $a['order'] );
							$b = intval( $b['order'] );

							if ( $a === $b ) {
								return 0;
							}

							return ( $a < $b ) ? -1 : 1;
						}
					);
					?>
					<address class="th-footer-contacts__contacts">
						<?php foreach ( $contacts as $contact ) : ?>
							<?php
							$caption = ! empty( $contact['caption'] ) ? $contact['caption'] : '';
							$value   = ! empty( $contact['value'] ) ? $contact['value'] : '';
							?>
							<dl>
								<dt><?php echo esc_html( $caption ); ?></dt>
								<dd><?php echo wp_kses( $value, 'redparts_sputnik_text' ); ?></dd>
							</dl>
						<?php endforeach; ?>
					</address>
				<?php endif; ?>
			</div>
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'     => 'show_logo',
					'label'   => esc_html__( 'Show logo instead of title', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
					),
				),
				array(
					'key'   => 'logo',
					'label' => esc_html__( 'Logo', 'redparts-sputnik' ),
					'type'  => 'image',
				),
				array(
					'key'          => 'text',
					'label'        => esc_html__( 'Text', 'redparts-sputnik' ),
					'type'         => 'textarea',
					'translatable' => true,
				),
				array(
					'key'    => 'contacts',
					'label'  => esc_html__( 'Contacts', 'redparts-sputnik' ),
					'type'   => 'tabs',
					'fields' => array(
						array(
							'key'   => 'order',
							'label' => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'  => 'number',
						),
						array(
							'key'          => 'caption',
							'label'        => esc_html__( 'Caption', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'value',
							'label'        => esc_html__( 'Value', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
					),
					'l10n'   => array(
						'add'    => esc_html__( 'Add Contact', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Contact', 'redparts-sputnik' ),
						/* translators: %s: Contact number. */
						'title'  => esc_html__( 'Contact %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );
		}
	}
}
