<?php
/**
 * Plugin Name: RedParts Sputnik
 * Description: Satellite plugin for the RedParts Theme.
 * Version: 1.15.0
 * Author: Konstantin Veselovsky
 * Author URI: https://themeforest.net/user/kos9
 * License: ThemeForest Split License
 * License URI: http://themeforest.net/licenses/standard
 * Text Domain: redparts-sputnik
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 **/

use RedParts\Sputnik\Plugin as RedPartsSputnik;

defined( 'ABSPATH' ) || exit;

define( 'RED_PARTS_SPUTNIK_FILE', __FILE__ );

if ( ! function_exists( 'redparts_sputnik_autoload' ) ) {
	/**
	 * Automatically loads RedParts Sputnik classes.
	 *
	 * @param string $classname - Class name.
	 * @noinspection DuplicatedCode
	 */
	function redparts_sputnik_autoload( string $classname ) {
		$prefix = 'RedParts\\Sputnik\\';

		if ( substr( $classname, 0, strlen( $prefix ) ) !== $prefix ) {
			return;
		}

		$classname = substr( $classname, strlen( $prefix ) );
		$classname = strtolower( str_replace( '_', '-', $classname ) );
		$parts     = explode( '\\', $classname );

		if ( 1 > count( $parts ) ) {
			return;
		}

		$parts[ count( $parts ) - 1 ] = 'class-' . $parts[ count( $parts ) - 1 ] . '.php';

		$filename = plugin_dir_path( __FILE__ ) . 'inc' . DIRECTORY_SEPARATOR . implode( DIRECTORY_SEPARATOR, $parts );

		if ( ! file_exists( $filename ) ) {
			return;
		}

		/** Redundant inspection. @noinspection PhpIncludeInspection */
		require_once $filename;
	}

	spl_autoload_register( 'redparts_sputnik_autoload' );
}

if ( ! function_exists( 'redparts_sputnik_get_classes' ) ) {
	/**
	 * Combines CSS classes.
	 *
	 * @noinspection DuplicatedCode
	 *
	 * @param string|array ...$classes Array of CSS classes.
	 * @return string
	 */
	function redparts_sputnik_get_classes( ...$classes ): string {
		$result = array();

		foreach ( $classes as $class ) {
			if ( is_string( $class ) && trim( $class ) ) {
				$result[] = trim( $class );
			} elseif ( is_array( $class ) ) {
				foreach ( $class as $subclass => $condition ) {
					if ( $condition && trim( $subclass ) ) {
						$result[] = trim( $subclass );
					}
				}
			}
		}

		return implode( ' ', $result );
	}
}

if ( ! function_exists( 'redparts_sputnik_the_classes' ) ) {
	/**
	 * Combines and outputs CSS classes.
	 *
	 * @param array ...$classes Array of CSS classes.
	 */
	function redparts_sputnik_the_classes( ...$classes ) {
		echo esc_attr( redparts_sputnik_get_classes( ...$classes ) );
	}
}

if ( ! function_exists( 'redparts_sputnik_image_allowed_html' ) ) {
	/**
	 * Returns image allowed html.
	 */
	function redparts_sputnik_image_allowed_html(): array {
		return apply_filters(
			'redparts_sputnik_image_allowed_html',
			array(
				'img' => array(
					'src'     => true,
					'width'   => true,
					'height'  => true,
					'class'   => true,
					'alt'     => true,
					'loading' => true,
				),
			)
		);
	}
}


if ( ! function_exists( 'redparts_sputnik_kses_allowed_html' ) ) {
	/**
	 * Returns allowed html tags.
	 *
	 * @param array  $tags    - Array of allowed tags.
	 * @param string $context - Context.
	 *
	 * @return array
	 */
	function redparts_sputnik_kses_allowed_html( array $tags, string $context ): array {
		switch ( $context ) {
			case 'redparts_sputnik_block_slideshow_title':
			case 'redparts_sputnik_block_slideshow_details':
			case 'redparts_sputnik_block_banner_title':
			case 'redparts_sputnik_block_banner_subtitle':
			case 'redparts_sputnik_block_teammates_title':
			case 'redparts_sputnik_block_teammates_subtitle':
			case 'redparts_sputnik_block_reviews_title':
			case 'redparts_sputnik_block_reviews_subtitle':
				return array(
					'br' => array(),
				);
			case 'redparts_sputnik_text':
				return array(
					'br' => array(),
					'a'  => array(
						'href' => true,
					),
				);
			case 'redparts_sputnik_stock':
				return array(
					'div' => array(
						'class' => true,
					),
				);
			case 'redparts_sputnik_image':
				return array(
					'img' => array(
						'src'     => true,
						'width'   => true,
						'height'  => true,
						'class'   => true,
						'alt'     => true,
						'loading' => true,
					),
				);
			case 'redparts_sputnik_categories_list':
				return array(
					'a' => array(
						'href' => true,
						'rel'  => true,
					),
				);
			case 'redparts_sputnik_post_card_date':
				return array(
					'time' => array(
						'class'    => array(),
						'datetime' => array(),
					),
					'a'    => array(
						'href'  => array(),
						'title' => array(),
						'rel'   => array(),
					),
				);
			case 'redparts_sputnik_star_rating':
				return array(
					'span'   => array(
						'class' => true,
						'style' => true,
					),
					'strong' => array(
						'class' => true,
					),
				);
			case 'redparts_sputnik_svg':
				return array(
					'svg'    => array(
						'class'   => true,
						'xmlns'   => true,
						'viewbox' => true,
						'width'   => true,
						'height'  => true,
					),
					'path'   => array(
						'd' => true,
					),
					'circle' => array(
						'cx' => true,
						'cy' => true,
						'r'  => true,
					),
				);
			default:
				return $tags;
		}
	}

	add_filter( 'wp_kses_allowed_html', 'redparts_sputnik_kses_allowed_html', 10, 2 );
}

if ( ! function_exists( 'redparts_sputnik' ) ) {
	/**
	 * Returns the main plugin object.
	 *
	 * @return RedPartsSputnik
	 */
	function redparts_sputnik(): RedPartsSputnik {
		return RedPartsSputnik::instance();
	}

	redparts_sputnik()->init();
}
