<?php

/**
 * An external standard for AuroShip.
 *
 * @package AuroShip
 * @link    https://www.aurobees.com/
 */
if ( !defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Required minimums and constants
 */
require_once AUROSHIP_PLUGIN_PATH . '/includes/class-auroship-api.php';
require_once AUROSHIP_PLUGIN_PATH . '/includes/functions.php';
require_once AUROSHIP_PLUGIN_PATH . '/includes/auroship-email.php';
require_once AUROSHIP_PLUGIN_PATH . '/includes/generate-label-server.php';
require_once AUROSHIP_PLUGIN_PATH . '/includes/tracking-shipment-server.php';
require_once AUROSHIP_PLUGIN_PATH . '/includes/tracking-shipment-details-server.php';

/**
 * WC_Auroship class for generating label functionality.
 *
 * @package AuroShip
 * @link    https://www.aurobees.com/
 */
class WC_Auroship {
	/**
	 * Class Constructor
	 */
	public function __construct() {
		add_action( 'admin_init', array( $this, 'check_order_edit_page' ) );
		add_action( 'woocommerce_view_order', array( $this, 'tracking_info_on_order_details' ) );
		add_action( 'dokan_order_detail_after_order_notes', array( $this, 'render_package_label_form_dokan' ) );
		add_action( 'dokan_order_detail_after_order_notes', function ( $order ) {
			$this->render_tracking_form( $order, 'dokan' );
		} );
	}

	/**
	 * Check if it's the WooCommerce admin order details page
	 *
	 * @param int $order_id Unique order ID.
	 * @return array
	 */
	public function check_order_edit_page() {
		$order_id = null; // Initialize order ID variable

		// Check for the new WooCommerce Orders screen URL format
		if ( isset( $_GET['page'], $_GET['action'], $_GET['id'] )
			&& 'wc-orders' === sanitize_text_field( $_GET['page'] )
			&& 'edit' === sanitize_text_field( $_GET['action'] ) ) {
			$order_id = intval( $_GET['id'] );
		}
		// Check for the traditional WordPress post editing URL format
		elseif ( isset( $_GET['post'], $_GET['action'] )
			&& 'edit' === sanitize_text_field( $_GET['action'] )
			&& 'shop_order' === get_post_type( intval( $_GET['post'] ) ) ) {
			$order_id = intval( $_GET['post'] );
		}

		if ( $order_id ) {
			add_action( 'add_meta_boxes', array( $this, 'register_package_label_menu' ) );
			add_action( 'add_meta_boxes', array( $this, 'register_tracking_menu' ) );
		}
	}

	/**
	 * Create request param to get package label.
	 *
	 * @param int $order_id Unique order ID.
	 * @return array
	 */
	public function get_label_generate_params( $order_id ) {
		$order = wc_get_order( $order_id );
		return $this->get_package_details( $order );
	}

	/**
	 * Get ship-to address.
	 *
	 * @param array $order Order array.
	 * @return array
	 */
	public function get_shipping_address( $order, $order_id ) {
		$first_name = $order->get_shipping_first_name();
		if ( empty( $first_name ) ) {
			$first_name = $order->get_billing_first_name();
		}

		$last_name = $order->get_shipping_last_name();
		if ( empty( $last_name ) ) {
			$last_name = $order->get_billing_last_name();
		}

		$address1 = $order->get_shipping_address_1();
		if ( empty( $address1 ) ) {
			$address1 = $order->get_billing_address_1();
		}
		if ( empty( $address1 ) ) {
			$address1 = "";
		}

		$address2 = $order->get_shipping_address_2();
		if ( empty( $address2 ) ) {
			$address2 = $order->get_billing_address_2();
		}
		if ( empty( $address2 ) ) {
			$address2 = "";
		}

		$company_name = $order->get_shipping_company();
		if ( empty( $company_name ) ) {
			$company_name = $order->get_billing_company();
		}
		if ( empty( $company_name ) ) {
			$company_name = "";
		}

		$state = $order->get_shipping_state();
		if ( empty( $state ) ) {
			$state = $order->get_billing_state();
		}

		$city = $order->get_shipping_city();
		if ( empty( $city ) ) {
			$city = $order->get_billing_city();
		}

		$country = $order->get_shipping_country();
		if ( empty( $country ) ) {
			$country = $order->get_billing_country();
		}
		if ( empty( $country ) ) {
			$country = 'IN';
		}

		$postcode = $order->get_shipping_postcode();
		if ( empty( $postcode ) ) {
			$postcode = $order->get_billing_postcode();
		}

		$email = $order->get_billing_email();
		$phone = $order->get_billing_phone();

		update_post_meta( $order_id, '_buyer_fname', $first_name );
		update_post_meta( $order_id, '_buyer_email', $email );

		return array(
			'name' => $first_name . ' ' . $last_name,
			'addressLine1' => Auroship_utilities::trim_address_line( $address1 ),
			'addressLine2' => Auroship_utilities::trim_address_line( $address2 ),
			'addressLine3' => "", // Custom field if any
			'companyName' => $company_name,
			'stateOrRegion' => $state,
			'city' => $city,
			'countryCode' => $country,
			'postalCode' => $postcode,
			'email' => $email,
			'phoneNumber' => $phone,
		);
	}

	/**
	 * Get ship-from address.
	 *
	 * @param array $seller Seller array.
	 * @return array
	 */
	public function get_ship_from_address( $seller, $order_id ) {
		if ( empty( $seller ) ) {
			$admin_email = get_option( 'admin_email' );
			$admin_user = get_user_by( 'email', $admin_email );

			if ( !empty( $admin_user ) ) {
				$admin_user_id = $admin_user->ID;
				$admin_first_name = get_user_meta( $admin_user_id, 'first_name', true );
				$admin_last_name = get_user_meta( $admin_user_id, 'last_name', true );
				$admin_phone_number = get_user_meta( $admin_user_id, 'billing_phone', true );
				$admin_phone = !empty( $admin_phone_number ) ? $admin_phone_number : "";
			} else {
				Auroship_utilities::log( 'No admin user found.' );
			}
			$street_1 = !empty( get_option( 'woocommerce_store_address' ) ) ? get_option( 'woocommerce_store_address' ) : "";
			$street_2 = !empty( get_option( 'woocommerce_store_address_2' ) ) ? get_option( 'woocommerce_store_address_2' ) : "";
			$city = !empty( get_option( 'woocommerce_store_city' ) ) ? get_option( 'woocommerce_store_city' ) : "";
			$postcode = !empty( get_option( 'woocommerce_store_postcode' ) ) ? get_option( 'woocommerce_store_postcode' ) : "";
			$store_raw_country = get_option( 'woocommerce_default_country' );
			$split_country = explode( ':', $store_raw_country );
			$store_country = isset( $split_country[0] ) ? $split_country[0] : "";
			$store_state = isset( $split_country[1] ) ? $split_country[1] : "";
			$country = !empty( $store_country ) ? $store_country : "";
			$state = !empty( $store_state ) ? $store_state : "";
			$company = !empty( get_option( 'blogname' ) ) ? get_option( 'blogname' ) : "";
			$email = !empty( $admin_email ) ? $admin_email : "";
			$phone = !empty( $admin_phone ) ? $admin_phone : "";
			$name = $admin_first_name . ' ' . $admin_last_name;
		} else {
			$street_1 = !empty( $seller->get_address()['street_1'] ) ? $seller->get_address()['street_1'] : get_option( 'woocommerce_store_address_1' );
			$street_2 = !empty( $seller->get_address()['street_2'] ) ? $seller->get_address()['street_2'] : get_option( 'woocommerce_store_address_2' );
			$city = !empty( $seller->get_address()['city'] ) ? $seller->get_address()['city'] : get_option( 'woocommerce_store_city' );
			$postcode = !empty( $seller->get_address()['zip'] ) ? $seller->get_address()['zip'] : get_option( 'woocommerce_store_postcode' );
			$store_raw_country = get_option( 'woocommerce_default_country' );
			$split_country = explode( ':', $store_raw_country );
			$store_country = isset( $split_country[0] ) ? $split_country[0] : "";
			$store_state = isset( $split_country[1] ) ? $split_country[1] : "";
			$country = !empty( $seller->get_address()['country'] ) ? $seller->get_address()['country'] : $store_country;
			$state = !empty( $seller->get_address()['state'] ) ? $seller->get_address()['state'] : $store_state;
			$company = !empty( $seller->get_shop_name() ) ? $seller->get_shop_name() : get_option( 'blogname' );
			$email = !empty( $seller->get_email() ) ? $seller->get_email() : "";
			$phone = !empty( $seller->get_phone() ) ? $seller->get_phone() : "";
			$first_name = !empty( $seller->get_first_name() ) ? $seller->get_first_name() : "";
			$last_name = !empty( $seller->get_last_name() ) ? $seller->get_last_name() : "";
			$name = $first_name . ' ' . $last_name;
		}

		update_post_meta( $order_id, '_seller_fname', $first_name );
		update_post_meta( $order_id, '_seller_email', $email );

		return array(
			'name' => $name,
			'addressLine1' => Auroship_utilities::trim_address_line( $street_1 ),
			'addressLine2' => Auroship_utilities::trim_address_line( $street_2 ),
			'addressLine3' => "",
			'companyName' => $company,
			'stateOrRegion' => $state,
			'city' => $city,
			'countryCode' => $country,
			'postalCode' => $postcode,
			'email' => $email,
			'phoneNumber' => $phone,
		);
	}

	/**
	 * Get package details.
	 *
	 * @param array $order Order array.
	 * @return array
	 */
	public function get_package_details( $order ) {
		$items_data = array();
		$product_desc = array();
		$product_sku = array();
		$product_qty = 0;
		$product_weight = 0;
		$vendor_info = array();
		$vendor_name = "";
		$cod_details = array();
		if ( $order ) {
			$items = $order->get_items();
			if ( empty( $items ) ) {
				echo 'No items found in the order.';
			} else {
				foreach ( $order->get_items() as $item ) {
					$product = $item->get_product();

					if ( is_a( $product, 'WC_Product' ) ) {
						$product_id = $product->get_id();
						if ( in_array( 'dokan-lite/dokan.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) && function_exists( 'dokan' ) ) {
							$vendor_id = get_post_field( 'post_author', $product_id );
							$vendor_info = dokan()->vendor->get( $vendor_id );
							$vendor_name = $vendor_info->get_shop_name();
						} else {
							$vendor_name = get_option( 'blogname' );
						}
						$product_sku[] = $product->get_sku();
						$product_desc[] = $product->get_name();
						$product_qty = 1;

						$weight_unit = get_option( 'woocommerce_weight_unit' );
						$product_weights = (float) $product->get_weight();
						$converted_weight = Auroship_utilities::convert_to_grams( $product_weights, $weight_unit );
						$product_weight += ( ( (float) $converted_weight ) * $item->get_quantity() );
					}
					update_post_meta( $order->get_id(), '_product_weight', $product_weight );

					$items_data = array(
						'itemValue' => array(
							'value' => (float) $order->get_total(),
							'unit' => $order->get_currency(),
						),
						'description' => substr( implode( ',', $product_desc ), 0, 400 ),
						'itemIdentifier' => implode( ',', $product_sku ),
						'quantity' => $product_qty,
						'weight' => array(
							'unit' => 'GRAM',
							'value' => $product_weight,
						),
						'isHazmat' => false,
						'productType' => $product->get_type(),
					);
				}
			}

			$package_details = array(
				'dimensions' => array(
					'length' => 0,
					'width' => 0,
					'height' => 0,
					'unit' => 'CENTIMETER',
				),
				'weight' => array(
					'unit' => 'GRAM',
					'value' => 0,
				),
				'insuredValue' => array(
					'value' => (float) $order->get_total(),
					'unit' => $order->get_currency(),
				),
				'isHazmat' => false,
				'sellerDisplayName' => $vendor_name,
				'packageClientReferenceId' => (string) $order->get_id(),
				'items' => [$items_data],
			);

			$shipping_address = $this->get_shipping_address( $order, $order->get_id() );

			$ship_from_address = $this->get_ship_from_address( $vendor_info, $order->get_id() );
			$return_to_address = $ship_from_address; // Assuming return address is the same as the ship from address

			if ( strtolower( $order->get_payment_method() ) == 'cod' ) {
				$cod_details = array(
					array(
						'id' => "CollectOnDelivery",
						'amount' => array(
							'unit' => $order->get_currency(),
							'value' => (float) $order->get_total(),
						),
					),
				);
			}
			return array(
				'shipTo' => $shipping_address,
				'shipFrom' => $ship_from_address,
				'returnTo' => $return_to_address,
				'packages' => array( $package_details ),
				'valueAddedServiceDetails' => $cod_details,
				'taxDetails' => array(
					array(
						'taxType' => 'GST',
						'taxRegistrationNumber' => get_option( 'auroship_tax_registration_number' ),
					),
				),
				'channelDetails' => array(
					'channelType' => 'EXTERNAL',
				),
				'serviceSelection' => array(
					'serviceId' => array( get_option( 'auroship_amzn_service_id' ) ),
				),
				'labelSpecifications' => array(
					'format' => 'PDF',
					'size' => array(
						'width' => 4,
						'length' => 6,
						'unit' => 'INCH',
					),
					'dpi' => 203,
					'pageLayout' => 'DEFAULT',
					'needFileJoining' => false,
					'requestedDocumentTypes' => array( 'LABEL' ),
				),
			);
		}
	}

	/**
	 * Register the package label menu
	 *
	 * @return void
	 */
	public function register_package_label_menu() {
		add_meta_box(
			'Package label for shipping',
			esc_html__( 'Packaging Label', 'auroship' ),
			[$this, 'render_package_label_form'],
			"",
			'side',
			'high'
		);
	}

	/**
	 * Register tracking menu
	 *
	 * @return void
	 */
	public function register_tracking_menu() {
		add_meta_box(
			'Shipment Tracking',
			esc_html__( 'Shipment Tracking', 'auroship' ),
			[$this, 'render_tracking_form'],
			"",
			'side',
			'low'
		);
	}

	/**
	 * Tracking form
	 *
	 * @param WC_Order $order WooCommerce Order object.
	 * @return void
	 */
	public function render_tracking_form( $orders, $template_type = 'standard' ) {
		$data = $this->tracking_info( $orders );
		$template_path = 'dokan' === $template_type ? AUROSHIP_PLUGIN_PATH . '/templates/tracking-shipment-form-dokan.php'
		: AUROSHIP_PLUGIN_PATH . '/templates/tracking-shipment-form.php';

		if ( file_exists( $template_path ) ) {
			extract( $data );
			include_once $template_path;
		}
	}

	/**
	 * Package label form
	 *
	 * @param WC_Order $order WooCommerce Order object.
	 * @return void
	 */
	public function render_package_label_form() {
		$order_data = $this->get_order_data();
		$btn_disabled = "";
		if ( $order_data ) {
			if ( !empty( $order_data['error_message'] ) ) {
				echo esc_html( $order_data['error_message'] );
			} else {
				$orders = wc_get_order( $order_data['data']['order_id'] );
				$order_status = $orders->get_status();
				if ( 'confirmed' == strtolower( $order_status ) || 'processing' == strtolower( $order_status ) ) {
					$btn_disabled = "";
				} else {
					$btn_disabled = "disabled";
				}
				$order_data['data']['btn_disabled'] = $btn_disabled;
				$this->display_package_label_form( $order_data['data'] );
			}
		}
	}

	/**
	 * Package label form for dokan
	 *
	 * @param WC_Order $order WooCommerce Order object.
	 * @return void
	 */
	public function render_package_label_form_dokan( $order ) {
		$order_data = $this->get_order_data();
		$order_status_dokan = $order->get_status();
		$btn_disabled_dokan = "";

		if ( $order_data ) {
			if ( !empty( $order_data['error_message'] ) ) {
				echo esc_html( $order_data['error_message'] );
			} else {
				if ( 'confirmed' == $order_status_dokan || 'processing' == $order_status_dokan ) {
					$btn_disabled_dokan = "";
					if ( !empty( $order_data['data']['tracking_id'] ) ) {
						$btn_disabled_dokan = "disabled";
					}
				} else {
					$btn_disabled_dokan = "disabled";
				}

				$order_data['data']['btn_disabled_dokan'] = $btn_disabled_dokan;
				$this->display_package_label_form_dokan( $order_data['data'] );
			}
		}
	}

	/**
	 * Get the order data from order ID of the URL.
	 *
	 * @return array The order ID.
	 */
	private function get_order_data( $order_id_param = 'order_id' ) {
		$order_id = 0;
		$error_message = "";
		$data = null;

		if ( is_admin() ) {
			if ( isset( $_GET['post'] ) ) {
				$order_id = intval( $_GET['post'] );
			} elseif ( isset( $_GET['id'] ) ) {
				$order_id = intval( $_GET['id'] );
			}
		} else {
			if ( isset( $_GET[$order_id_param] ) ) {
				$order_id = intval( $_GET[$order_id_param] );
			}
		}

		if ( !$order_id ) {
			return false;
		} else {
			$order = wc_get_order( $order_id );

			if ( !$order ) {
				$error_message = esc_html__( 'Order not found.', 'auroship' );
			} else {
				$items = $order->get_items();
				if ( empty( $items ) ) {
					$error_message = esc_html__( 'No items found in the order.', 'auroship' );
				} else {
					$data = $this->calculate_package_data( $order );
				}
			}
		}

		return ['data' => $data, 'error_message' => $error_message];
	}

	/**
	 * Calculate the package data needed for the form.
	 *
	 * @param WC_Order $order WooCommerce Order object.
	 * @return array Calculated data.
	 */
	private function calculate_package_data( $order ) {
		$auroship_product_weight = 0;
		$max_total_lenwh = 0;
		$max_plength = 0;
		$max_pwidth = 0;
		$max_pheight = 0;

		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();

			if ( is_a( $product, 'WC_Product' ) ) {
				$weight_unit = get_option( 'woocommerce_weight_unit' );
				$dimension_unit = get_option( 'woocommerce_dimension_unit' );
				$product_weight = (float) $product->get_weight();
				$converted_weight = Auroship_utilities::convert_to_grams( $product_weight, $weight_unit );
				$auroship_product_weight += ( $converted_weight * $item->get_quantity() );

				$length_in_cm = (float) Auroship_utilities::convert_to_centimeters( $product->get_length(), $dimension_unit );
				$width_in_cm = (float) Auroship_utilities::convert_to_centimeters( $product->get_width(), $dimension_unit );
				$height_in_cm = (float) Auroship_utilities::convert_to_centimeters( $product->get_height(), $dimension_unit );

				if ( $length_in_cm > 0 && $width_in_cm > 0 && $height_in_cm > 0 ) {
					$total_lwh = $length_in_cm * $width_in_cm * $height_in_cm;

					if ( $total_lwh > $max_total_lenwh ) {
						$max_total_lenwh = $total_lwh;
						$max_plength = $length_in_cm;
						$max_pwidth = $width_in_cm;
						$max_pheight = $height_in_cm;
					}
				}
			}
		}

		$order_id = $order->get_id();
		$product_weights = get_post_meta( $order_id, '_auroship_product_weight', true );
		$package_len = get_post_meta( $order_id, '_package_length', true );
		$package_wid = get_post_meta( $order_id, '_package_width', true );
		$package_hght = get_post_meta( $order_id, '_package_height', true );
		$tracking_id = get_post_meta( $order_id, '_tracking_id', true );

		$pdf_url = get_post_meta( $order_id, '_azure_pdf_file_url', true );
		if ( empty( $pdf_url ) || !class_exists( 'WindowsAzureStorageUtil' ) ) {
			$pdf_url = get_post_meta( $order_id, '_pdf_file_url', true );
		}

		return [
			'order_id' => $order_id,
			'product_weight' => !empty( $product_weights ) ? $product_weights : $auroship_product_weight,
			'package_length' => !empty( $package_len ) ? $package_len : $max_plength,
			'package_width' => !empty( $package_wid ) ? $package_wid : $max_pwidth,
			'package_height' => !empty( $package_hght ) ? $package_hght : $max_pheight,
			'tracking_id' => $tracking_id,
			'pdf_url' => $pdf_url,
			'btntext' => !empty( $tracking_id ) ? esc_html__( 'Re-generate Label', 'auroship' ) : esc_html__( 'Generate Label', 'auroship' ),
			'awbtext' => !empty( $tracking_id ) ? '<span>' . esc_html__( 'AWB Number:', 'auroship' ) . '</span> <span><strong>' . esc_html( $tracking_id ) . '</strong></span>' : "",
			'pdfurltext' => !empty( $pdf_url ) ? '<strong><a class="blue-color" download id="package-label-link" href="' . esc_url( $pdf_url ) . '">' . esc_html__( 'Download Label', 'auroship' ) . '</a></strong>' : "",
		];
	}

	/**
	 * Display the package label form.
	 *
	 * @param array $data Data for displaying the form.
	 * @return void
	 */
	private function display_package_label_form( $data ) {
		$template_path = AUROSHIP_PLUGIN_PATH . '/templates/package-label-form.php';

		if ( file_exists( $template_path ) ) {
			include_once $template_path;
		}
	}

	/**
	 * Display the package label form for dokan.
	 *
	 * @param array $data Data for displaying the form.
	 * @return void
	 */
	private function display_package_label_form_dokan( $data ) {
		$template_path = AUROSHIP_PLUGIN_PATH . '/templates/package-label-form-dokan.php';

		if ( file_exists( $template_path ) ) {
			include_once $template_path;
		}
	}

	public function tracking_info( $orders ) {
		$order = wc_get_order( $orders );
		if ( !$order ) {
			return false;
		}

		$track_id = get_post_meta( $order->get_id(), '_tracking_id', true );
		$tracking_id = !empty( $track_id ) ? $track_id : '';
		$carrier_id = get_post_meta( $order->get_id(), '_carrier_id', true );
		$shipping_provider = !empty( get_post_meta( $order->get_id(), '_shipping_provider', true ) ) ? get_post_meta( $order->get_id(), '_shipping_provider', true ) : "";
		$shipping_status = get_post_meta( $order->get_id(), '_shipping_status', true );
		$est_delivery_date = get_post_meta( $order->get_id(), '_promised_delivery_date', true );
		$tracking_history = get_post_meta( $order->get_id(), '_tracking_history', true );
		$tracking_history_arr = json_decode( $tracking_history );
		if ( !empty( $est_delivery_date ) && strtotime( $est_delivery_date ) !== false ) {
			$formatted_date = new DateTime( $est_delivery_date );
			$estimated_delivery_date = $formatted_date->format( 'd-m-Y' );
		} else {
			$estimated_delivery_date = '';
		}
		$tracking_event_code = [
			'ReadyForReceive', 'PickupDone', 'Delivered', 'Departed', 'DeliveryAttempted', 'Lost',
			'OutForDelivery', 'ArrivedAtCarrierFacility', 'Rejected', 'Undeliverable',
			'PickupCancelled', 'ReturnInitiated', 'AvailableForPickup',
		];

		$specific_statuses = ['PickupCancelled', 'ReturnInitiated', 'Rejected', 'Undeliverable', 'Lost', 'Delivered'];
		$btn_disabled_tracking_details = "";
		if ( 'delivered' == strtolower( $order->get_status() ) ) {
			$btn_disabled_tracking_details = 'disabled';
		}
		return [
			'tracking_id' => $tracking_id,
			'carrier_id' => $carrier_id,
			'shipping_provider' => $shipping_provider,
			'shipping_status' => $shipping_status,
			'estimated_delivery_date' => $estimated_delivery_date,
			'tracking_history_arr' => $tracking_history_arr,
			'tracking_event_code' => $tracking_event_code,
			'specific_statuses' => $specific_statuses,
			'btn_disabled_tracking_details' => $btn_disabled_tracking_details,
		];
	}

	/**
	 * Tracking form on order view page
	 *
	 * @param WC_Order $order WooCommerce Order object.
	 * @return void
	 */
	public function tracking_info_on_order_details( $order ) {
		$data = $this->tracking_info( $order );
		$template_path = AUROSHIP_PLUGIN_PATH . '/templates/tracking-shipment-details.php';
		if ( file_exists( $template_path ) ) {
			wp_enqueue_style( 'auroship_styles', AUROSHIP_PLUGIN_URL . '/assets/css/auroship-style.css', array(), AUROSHIP_PLUGIN_VERSION );
			extract( $data );
			include_once $template_path;
		}
	}

}

$auroship = new WC_Auroship();
