<?php

/**
 * Setup Custom Order Statuses for WooCommerce and manage Dokan compatibility.
 */
function auroship_setup_custom_order_statuses() {

	defined( 'AUROSHIP_ORDER_STATUS_CONTEXT' ) || define( 'AUROSHIP_ORDER_STATUS_CONTEXT', 'Order status' );

	// Define custom statuses.
	$statuses = [
		'confirmed' => _x( 'Confirmed', AUROSHIP_ORDER_STATUS_CONTEXT, 'auroship' ),
		'shipped' => _x( 'Shipped', AUROSHIP_ORDER_STATUS_CONTEXT, 'auroship' ),
		'delivered' => _x( 'Delivered', AUROSHIP_ORDER_STATUS_CONTEXT, 'auroship' ),
	];

	// Register each custom status.
	foreach ( $statuses as $status_key => $status_label ) {
		register_post_status(
			'wc-' . $status_key,
			array(
				'label' => $status_label,
				'public' => true,
				'exclude_from_search' => false,
				'show_in_admin_all_list' => true,
				'show_in_admin_status_list' => true,
				'label_count' => _n_noop( $status_label . ' (%s)', $status_label . ' (%s)', 'auroship' ),
			)
		);
	}

	// Add custom statuses to WooCommerce order statuses.
	add_filter( 'wc_order_statuses', function ( $order_statuses ) use ( $statuses ) {
		foreach ( $statuses as $status_key => $status_label ) {
			$order_statuses['wc-' . $status_key] = $status_label;
		}
		return $order_statuses;
	} );

	// Add custom statuses to bulk actions and handle the bulk actions.
	add_filter( 'bulk_actions-edit-shop_order', function ( $bulk_actions ) {
		$bulk_actions['mark_confirmed'] = __( 'Change status to Confirmed', 'auroship' );
		$bulk_actions['mark_shipped'] = __( 'Change status to Shipped', 'auroship' );
		$bulk_actions['mark_delivered'] = __( 'Change status to Delivered', 'auroship' );
		return $bulk_actions;
	} );

	add_filter( 'handle_bulk_actions-edit-shop_order', function ( $redirect_to, $action, $post_ids ) {
		$statuses = [
			'mark_confirmed' => 'confirmed',
			'mark_shipped' => 'shipped',
			'mark_delivered' => 'delivered',
		];

		if ( array_key_exists( $action, $statuses ) ) {
			foreach ( $post_ids as $post_id ) {
				$order = wc_get_order( $post_id );
				$order->update_status( 'wc-' . $statuses[$action] );
			}
			$redirect_to = add_query_arg( $action . '_count', count( $post_ids ), $redirect_to );
		}
		return $redirect_to;
	}, 10, 3 );

	// Add order note when status is set to a custom status.
	foreach ( ['confirmed', 'shipped', 'delivered'] as $status_key ) {
		add_action( 'woocommerce_order_status_wc-' . $status_key, function ( $order_id ) use ( $status_key ) {
			$order = wc_get_order( $order_id );
			$order->add_order_note( __( 'Order has been ' . $status_key . '.', 'auroship' ) );
		} );
	}

	// Ensure custom statuses are included in WooCommerce admin order queries.
	add_filter( 'woocommerce_admin_order_statuses', function ( $order_statuses ) {
		return array_merge( $order_statuses, [
			'wc-confirmed' => _x( 'Confirmed', AUROSHIP_ORDER_STATUS_CONTEXT, 'auroship' ),
			'wc-shipped' => _x( 'Shipped', AUROSHIP_ORDER_STATUS_CONTEXT, 'auroship' ),
			'wc-delivered' => _x( 'Delivered', AUROSHIP_ORDER_STATUS_CONTEXT, 'auroship' ),
		] );
	} );

	// Ensure custom statuses are included in order reports.
	add_filter( 'woocommerce_reports_order_statuses', function ( $order_statuses ) {
		if ( is_array( $order_statuses ) ) {
			return array_merge( $order_statuses, [
				'wc-confirmed',
				'wc-shipped',
				'wc-delivered',
			] );
		}
		return $order_statuses;
	} );

	// Manage custom order statuses for Dokan if available.
	if ( function_exists( 'dokan' ) ) {
		add_filter( 'dokan_get_order_status_class', function ( $text, $status ) {
			$new_text = '';
			switch ( $status ) {
				case 'wc-confirmed':
				case 'confirmed':
					$new_text = 'info';
					break;
				case 'wc-shipped':
				case 'shipped':
					$new_text = 'warning';
					break;
				case 'wc-delivered':
				case 'delivered':
					$new_text = 'success';
					break;
				default:
					$new_text = 'default-class';
					break;
			}
			return $new_text;
		}, 10, 2 );

		add_filter( 'dokan_get_order_status_translated', function ( $text, $status ) {
			$new_text = '';
			switch ( $status ) {
				case 'wc-confirmed':
				case 'confirmed':
					$new_text = __( 'Confirmed', 'auroship' );
					break;
				case 'wc-shipped':
				case 'shipped':
					$new_text = __( 'Shipped', 'auroship' );
					break;
				case 'wc-delivered':
				case 'delivered':
					$new_text = __( 'Delivered', 'auroship' );
					break;
				default:
					$new_text = __( 'Unknown status', 'auroship' );
					break;
			}
			return $new_text;
		}, 10, 2 );
	}
}

add_action( 'init', 'auroship_setup_custom_order_statuses' );

add_action( 'template_redirect', 'auroship_check_if_dokan_dashboard' );

/**
 * Check the dokan dashboard
 *
 * @return void
 */
function auroship_check_if_dokan_dashboard() {
	$is_seller_dashboard = function_exists( 'dokan_is_seller_dashboard' ) && dokan_is_seller_dashboard();
	if ( 1 == $is_seller_dashboard ) {
		$order_id = isset( $_GET['order_id'] ) ? absint( $_GET['order_id'] ) : null;
		if ( $order_id ) {
			update_post_meta( $order_id, AUROSHIP_IS_SELLER_DASHBOARD, $is_seller_dashboard );
		}
	}
}

add_action( 'woocommerce_order_status_shipped', 'auroship_custom_status_notification', 20 );
add_action( 'woocommerce_order_status_confirmed', 'auroship_custom_status_notification', 20 );
add_action( 'woocommerce_order_status_delivered', 'auroship_custom_status_notification', 20 );

/**
 * Send mail for custom status
 *
 * @return void
 */
function auroship_custom_status_notification( $order_id ) {
	$order = wc_get_order( $order_id );
	$status = $order->get_status();
	$subject = '';
	$heading = '';

	$order_date = wc_format_datetime( $order->get_date_created() );
	$order_number = $order->get_order_number();

	switch ( $status ) {
		case 'shipped':
			$subject = "Your order has been shipped! #{$order_number} - {$order_date}";
			$subject_dokan = "Your customer order is now shipped! #{$order_number} - {$order_date}";
			$heading = "Shipped Customer Order #{$order_number}";
			break;
		case 'confirmed':
			$subject = "Your order is confirmed! #{$order_number} - {$order_date}";
			$subject_dokan = "Your customer order is now confirmed! #{$order_number} - {$order_date}";
			$heading = "Confirmed Customer Order #{$order_number}";
			break;
		case 'delivered':
			$subject = "Your order has been delivered! #{$order_number} - {$order_date}";
			$subject_dokan = "Your customer order is now delivered! #{$order_number} - {$order_date}";
			$heading = "Delivered Customer Order #{$order_number}";
			break;
		default:
			return;
	}

	$mailer = WC()->mailer()->get_emails();
	$mailer['WC_Email_Customer_Completed_Order']->settings['heading'] = $heading;
	$mailer['WC_Email_Customer_Completed_Order']->settings['subject'] = $subject;
	$mailer['WC_Email_Customer_Completed_Order']->trigger( $order_id );

	$is_seller_dashboard = get_post_meta( $order_id, AUROSHIP_IS_SELLER_DASHBOARD, true );

	if ( $is_seller_dashboard ) {
		$dokan_completed_order_email = $mailer['Dokan_Email_Completed_Order'];
		$dokan_completed_order_email->settings['heading'] = $heading;
		$dokan_completed_order_email->settings['subject'] = $subject_dokan;
		$dokan_completed_order_email->trigger( $order_id );
		update_post_meta( $order_id, AUROSHIP_IS_SELLER_DASHBOARD, 0 );
	}
}
