<?php

/**
 * An external standard for AuroShip.
 *
 * @package AuroShip
 * @link    https://www.aurobees.com/
 */

if ( !defined( 'ABSPATH' ) ) {
	exit;
}

require_once AUROSHIP_PLUGIN_PATH . '/vendor/autoload.php';

use MicrosoftAzure\Storage\Blob\BlobRestProxy;
use MicrosoftAzure\Storage\Common\Exceptions\ServiceException;

/**
 * Save the error logs in woocommerce
 *
 * @package AuroShip
 * @link    https://www.aurobees.com/
 */
class Auroship_utilities {
	/**
	 * Log
	 *
	 * @param $message error message
	 * @param string $level   information of error log
	 *
	 * @return void
	 */
	public static function log( $message, $level = 'info' ) {
		$options = get_option( 'woocommerce_auropay_settings' );
		$logging = $options['logging'];

		if ( 'no' != $logging ) {
			$log = wc_get_logger();
			$log->log( $level, $message, array( 'source' => AUROSHIP_PLUGIN_NAME ) );
		}
	}

	/**
	 * Convert base64 string to PDF and return the file URL
	 *
	 * @param string $base64_string Base64 encoded string.
	 * @param string $order_id unique id.
	 *
	 * @return string
	 */
	public static function auroship_decode_and_save_pdf( $base64_string, $order_id ) {
		$upload_dir = wp_upload_dir();
		$upload_path = $upload_dir['basedir'] . '/auroship';

		if ( !file_exists( $upload_path ) ) {
			wp_mkdir_p( $upload_path );
		}

		$sanitized_file_name = sanitize_file_name( 'Order-' . $order_id . '-Packing-Label.pdf' );

		$file_path = $upload_path . '/' . $sanitized_file_name;
		$pdf_data = base64_decode( $base64_string );

		if ( !$pdf_data || file_put_contents( $file_path, $pdf_data ) === false ) {
			return 'Error: Failed to decode or save the PDF file.';
		}

		$file_url = $upload_dir['baseurl'] . '/auroship/' . $sanitized_file_name;
		update_post_meta( $order_id, '_pdf_file_url', esc_url_raw( $file_url ) );
		if ( in_array( 'windows-azure-storage-4.3.5/windows-azure-storage.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
			$blob_name = 'auroship/' . $sanitized_file_name;
			$file_url = self::upload_pdf_to_azure_server( $file_url, $blob_name );
			update_post_meta( $order_id, '_azure_pdf_file_url', esc_url_raw( $file_url ) );
		}

		return $file_url;
	}

	/**
	 * Uploads a file to Azure Blob Storage.
	 *
	 * @param string $file_path Path to the file to upload.
	 * @param string $blob_name Name of the blob in Azure Storage.
	 * @return string Result return the file url.
	 */
	public static function upload_pdf_to_azure_server( $file_path, $blob_name ) {
		// Retrieve the Azure Storage settings from the Azure plugin
		$azur_account_name = get_option( 'azure_storage_account_name' );
		$azur_container_name = get_option( 'default_azure_storage_account_container_name' );
		$azur_access_key = get_option( 'azure_storage_account_primary_access_key' );

		// Initialize result variable
		$result = '';

		// Validate settings and file
		if ( empty( $azur_account_name ) || empty( $azur_access_key ) || empty( $azur_container_name ) ) {
			$result = __( 'Azure Storage settings are not configured properly.', 'auroship' );
			self::log( 'Azure Storage settings are not configured properly.' . $file_path );

		} else {
			try {
				// Create the Blob client
				$connectionString = "DefaultEndpointsProtocol=https;AccountName=$azur_account_name;AccountKey=$azur_access_key";
				$blobClient = BlobRestProxy::createBlobService( $connectionString );

				// Attempt to open the file
				$content = fopen( $file_path, 'r' );
				if ( false === $content ) {
					$result = __( 'Failed to open file. Please check the file path and permissions.', 'auroship' );
					self::log( 'Failed to open file. Please check the file path and permissions.' . $file_path );
				} else {
					// Upload the file as a blob
					$blobClient->createBlockBlob( $azur_container_name, $blob_name, $content );
					self::log( 'File uploaded successfully!' . $file_path );
					$result = "https://$azur_account_name.blob.core.windows.net/$azur_container_name/$blob_name";
				}
			} catch ( ServiceException $e ) {
				$result = sprintf( __( 'Error: %s - %s', 'auroship' ), $e->getCode(), $e->getMessage() );
			} catch ( Exception $e ) {
				$result = sprintf( __( 'General error: %s', 'auroship' ), $e->getMessage() );
			}
		}

		// Ensure the file handle is closed if it is still open
		if ( isset( $content ) && is_resource( $content ) ) {
			fclose( $content );
		}

		return esc_html( $result );
	}

	/**
	 * Convert the weight into grams
	 *
	 * @param string $weight weight of product.
	 * @param string $unit unit.
	 *
	 * @return float
	 */
	public static function convert_to_grams( $weight, $unit ) {
		switch ( strtolower( $unit ) ) {
			case 'kg':
				return $weight * 1000; // Convert kg to g
			case 'lbs':
				return $weight * 453.592; // Convert lbs to g
			case 'oz':
				return $weight * 28.3495; // Convert oz to g
			default:
				return $weight; // If already in grams or unrecognized unit, return the original weight
		}
	}

	/**
	 * Convert the length into centimeters
	 *
	 * @param string $length length of package.
	 * @param string $unit unit.
	 *
	 * @return float
	 */
	public static function convert_to_centimeters( $length, $from_unit ) {
		switch ( strtolower( $from_unit ) ) {
			case 'm': // Meters to Centimeters
				return $length * 100;
			case 'mm': // Millimeters to Centimeters
				return $length * 0.1;
			case 'in': // Inches to Centimeters
				return $length * 2.54;
			case 'yd': // Yards to Centimeters
				return $length * 91.44;
			default:
				return $length; // If the unit is already in centimeters or unrecognized, return the original length
		}
	}

	/**
	 * Trim the address to 60 characters without cutting off words.
	 *
	 * @param string $addressLine The original address.
	 * @return string The trimmed address.
	 *
	 * * @return string
	 */
	public static function trim_address_line( $addressLine ) {
		// Trim the address to a maximum of 60 characters
		$trimmed_address = substr( $addressLine, 0, 60 );

		// If the last character isn't a space and the address was trimmed, find the last space
		if ( strlen( $addressLine ) > 60 && substr( $trimmed_address, -1 ) !== ' ' ) {
			$trimmed_address = substr( $trimmed_address, 0, strrpos( $trimmed_address, ' ' ) );
		}

		return $trimmed_address;
	}
}
