<?php

/**
 * Setup Custom Order Statuses for WooCommerce and manage Dokan compatibility.
 */
function auroship_setup_custom_order_statuses() {
	// Define custom statuses.
	$statuses = [
		'confirmed' => _x( 'Confirmed', 'Order status', 'auroship' ),
		'shipped' => _x( 'Shipped', 'Order status', 'auroship' ),
		'delivered' => _x( 'Delivered', 'Order status', 'auroship' ),
	];

	// Register each custom status.
	foreach ( $statuses as $status_key => $status_label ) {
		register_post_status(
			'wc-' . $status_key,
			array(
				'label' => $status_label,
				'public' => true,
				'exclude_from_search' => false,
				'show_in_admin_all_list' => true,
				'show_in_admin_status_list' => true,
				'label_count' => _n_noop( $status_label . ' (%s)', $status_label . ' (%s)', 'auroship' ),
			)
		);
	}

	// Add custom statuses to WooCommerce order statuses.
	add_filter( 'wc_order_statuses', function ( $order_statuses ) use ( $statuses ) {
		foreach ( $statuses as $status_key => $status_label ) {
			$order_statuses['wc-' . $status_key] = $status_label;
		}
		return $order_statuses;
	} );

	// Add custom statuses to bulk actions and handle the bulk actions.
	add_filter( 'bulk_actions-edit-shop_order', function ( $bulk_actions ) {
		$bulk_actions['mark_confirmed'] = __( 'Change status to Confirmed', 'auroship' );
		$bulk_actions['mark_shipped'] = __( 'Change status to Shipped', 'auroship' );
		$bulk_actions['mark_delivered'] = __( 'Change status to Delivered', 'auroship' );
		return $bulk_actions;
	} );

	add_filter( 'handle_bulk_actions-edit-shop_order', function ( $redirect_to, $action, $post_ids ) {
		$statuses = [
			'mark_confirmed' => 'confirmed',
			'mark_shipped' => 'shipped',
			'mark_delivered' => 'delivered',
		];

		if ( array_key_exists( $action, $statuses ) ) {
			foreach ( $post_ids as $post_id ) {
				$order = wc_get_order( $post_id );
				$order->update_status( 'wc-' . $statuses[$action] );
			}
			$redirect_to = add_query_arg( $action . '_count', count( $post_ids ), $redirect_to );
		}
		return $redirect_to;
	}, 10, 3 );

	// Add order note when status is set to a custom status.
	foreach ( ['confirmed', 'shipped', 'delivered'] as $status_key ) {
		add_action( 'woocommerce_order_status_wc-' . $status_key, function ( $order_id ) use ( $status_key ) {
			$order = wc_get_order( $order_id );
			$order->add_order_note( __( 'Order has been ' . $status_key . '.', 'auroship' ) );
		} );
	}

	// Ensure custom statuses are included in WooCommerce admin order queries.
	add_filter( 'woocommerce_admin_order_statuses', function ( $order_statuses ) {
		return array_merge( $order_statuses, [
			'wc-confirmed' => _x( 'Confirmed', 'Order status', 'auroship' ),
			'wc-shipped' => _x( 'Shipped', 'Order status', 'auroship' ),
			'wc-delivered' => _x( 'Delivered', 'Order status', 'auroship' ),
		] );
	} );

	// Ensure custom statuses are included in order reports.
	add_filter( 'woocommerce_reports_order_statuses', function ( $order_statuses ) {
		if ( is_array( $order_statuses ) ) {
			return array_merge( $order_statuses, [
				'wc-confirmed',
				'wc-shipped',
				'wc-delivered',
			] );
		}
		return $order_statuses;
	} );

	// Manage custom order statuses for Dokan if available.
	if ( function_exists( 'dokan' ) ) {
		add_filter( 'dokan_get_order_status_class', function ( $text, $status ) {
			switch ( $status ) {
				case 'wc-confirmed':
				case 'confirmed':
					$text = 'info'; // Bootstrap class or any other custom class
					break;
				case 'wc-shipped':
				case 'shipped':
					$text = 'warning'; // Bootstrap class or any other custom class
					break;
				case 'wc-delivered':
				case 'delivered':
					$text = 'success'; // Bootstrap class or any other custom class
					break;
			}
			return $text;
		}, 10, 2 );

		add_filter( 'dokan_get_order_status_translated', function ( $text, $status ) {
			switch ( $status ) {
				case 'wc-confirmed':
				case 'confirmed':
					$text = __( 'Confirmed', 'auroship' );
					break;
				case 'wc-shipped':
				case 'shipped':
					$text = __( 'Shipped', 'auroship' );
					break;
				case 'wc-delivered':
				case 'delivered':
					$text = __( 'Delivered', 'auroship' );
					break;
			}
			return $text;
		}, 10, 2 );
	}
}

add_action( 'init', 'auroship_setup_custom_order_statuses' );

add_action( 'woocommerce_order_status_shipped', 'auroship_custom_status_notification', 20, 2 );
add_action( 'woocommerce_order_status_confirmed', 'auroship_custom_status_notification', 20, 2 );
add_action( 'woocommerce_order_status_delivered', 'auroship_custom_status_notification', 20, 2 );

function auroship_custom_status_notification( $order_id, $order ) {
	$order = wc_get_order( $order_id );
	$status = $order->get_status();
	$subject = '';
	$heading = '';

	$order_date = wc_format_datetime( $order->get_date_created() );
	$order_number = $order->get_order_number();

	switch ( $status ) {
		case 'shipped':
			$subject = "Your order has been shipped! #{$order_number} - {$order_date}";
			$heading = "Shipped Customer Order #{$order_number}";
			break;
		case 'confirmed':
			$subject = "Your order is confirmed! #{$order_number} - {$order_date}";
			$heading = "Confirmed Customer Order #{$order_number}";
			break;
		case 'delivered':
			$subject = "Your order has been delivered! #{$order_number} - {$order_date}";
			$heading = "Delivered Customer Order #{$order_number}";
			break;
		default:
			return;
	}

	$mailer = WC()->mailer()->get_emails();
	$mailer['WC_Email_Customer_Completed_Order']->settings['heading'] = $heading;
	$mailer['WC_Email_Customer_Completed_Order']->settings['subject'] = $subject;
	$mailer['WC_Email_Customer_Completed_Order']->trigger( $order_id );
}
